package com.xebialabs.xlrelease.triggers.scheduled.validators

import com.google.common.base.Strings.isNullOrEmpty
import com.xebialabs.deployit.plugin.api.validation.{ExtendedValidationContext, ValidationContext, Validator}
import com.xebialabs.xlrelease.domain.{PollType, ReleaseTrigger, ScheduledTrigger}
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.service.ConfigurationService
import com.xebialabs.xlrelease.support.akka.spring.ScalaSpringAwareBean
import com.xebialabs.xlrelease.triggers.scheduled.validators.ScheduledTriggerValidator.{NUMBER_OF_INTERVALS_TO_CHECK, configurationService}
import com.xebialabs.xlrelease.utils.Limits.{ENABLED, MIN_TRIGGER_INTERVAL, TYPE_LIMITS}
import com.xebialabs.xlrelease.utils.QuartzUtils._
import org.apache.commons.lang3.StringUtils
import org.apache.commons.lang3.math.NumberUtils
import org.joda.time.DateTime
import org.quartz.CronExpression

import java.util.Date
import scala.util.Try

object ScheduledTriggerValidator extends ScalaSpringAwareBean {
  val NUMBER_OF_INTERVALS_TO_CHECK = 5
  private lazy val configurationService: ConfigurationService = springBean[ConfigurationService]
}

class ScheduledTriggerValidator extends Validator[ScheduledTrigger] {

  override def validate(trigger: ScheduledTrigger, context: ValidationContext): Unit = {

    val limits = configurationService.getFeatureSettings(TYPE_LIMITS)
    val minTriggerInterval: Int = limits.getProperty(MIN_TRIGGER_INTERVAL)
    val limitsEnabled: Boolean = limits.getProperty(ENABLED)
    val extendedContext = context.asInstanceOf[ExtendedValidationContext]

    trigger.getPollType match {
      case PollType.REPEAT =>
        val period = NumberUtils.toInt(trigger.getPeriodicity, 0)
        if (period <= 0) {
          extendedContext.error(trigger, "periodicity", s"Trigger has wrong periodicity '${trigger.getPeriodicity}'")
        } else if (limitsEnabled && period < minTriggerInterval) {
          extendedContext.error(trigger, "periodicity",
            s"Trigger must not fire more frequently than once per $minTriggerInterval seconds")
        }

      case PollType.CRON =>
        val cron = trigger.getPeriodicity
        val errorMsg = s"Trigger has wrong cron expression '$cron'"
        if (StringUtils.isBlank(cron)) {
          extendedContext.error(trigger, "periodicity", errorMsg)
        } else {
          if (!CronExpression.isValidExpression(cron.asQuartzCron)) {
            extendedContext.error(trigger, "periodicity", errorMsg)
          } else if (limitsEnabled) {
            validateCron(cron, minTriggerInterval, trigger, extendedContext)
          }
        }
    }
  }

  private def validateCron(expression: String, limit: Int, trigger: ScheduledTrigger, extendedContext: ExtendedValidationContext): Unit = {
    val cronExpression = new CronExpression(expression.asQuartzCron)
    val today = new DateTime().withTimeAtStartOfDay

    val intervalsLongerThanLimit = areNextIntervalsShorterThanLimit(cronExpression, today.toDate, limit)

    if (!intervalsLongerThanLimit) {
      extendedContext.error(trigger, "periodicity",s"Trigger must not fire more frequently than once per $limit seconds")
    }
  }

  def areNextIntervalsShorterThanLimit(cronExpression: CronExpression, startTime: Date, limit: Int): Boolean = {
    Try {
      var reference = cronExpression.getNextValidTimeAfter(startTime)
      for (_ <- 0 to NUMBER_OF_INTERVALS_TO_CHECK) {
        val nextTime = cronExpression.getNextValidTimeAfter(reference)
        val interval = ((nextTime.getTime - reference.getTime) / 1000).toInt

        if (interval < limit) {
          // break, no need to loop further
          throw new RuntimeException()
        } else {
          // continue looping
          reference = nextTime
        }
      }
    }.isSuccess
  }
}

class ReleaseTriggerValidator extends Validator[ReleaseTrigger] {

  override def validate(trigger: ReleaseTrigger, context: ValidationContext): Unit = {
    val extendedContext = context.asInstanceOf[ExtendedValidationContext]
    if (isNullOrEmpty(trigger.getReleaseTitle)) {
      extendedContext.error(trigger, "releaseTitle", s"Release title cannot be blank")
    }

    validateTemplate(trigger, extendedContext)
  }
  private def validateTemplate(trigger: ReleaseTrigger, extendedContext: ExtendedValidationContext): Unit = {
    if (isNullOrEmpty(trigger.getTemplate)) {
      extendedContext.error(trigger, "template", s"Trigger does not have associated template")
    }
    if (!isNullOrEmpty(trigger.getTemplate)) {
      if (!isNullOrEmpty(trigger.getFolderId) && Ids.findFolderId(trigger.getTemplate) != trigger.getFolderId) {
        extendedContext.error(trigger, "template", s"Trigger folder id does not match template folder id")
      }
    }
  }
}