package com.xebialabs.xlrelease.triggers.scheduled.validators

import com.google.common.base.Strings.isNullOrEmpty
import com.xebialabs.deployit.plugin.api.validation.{ExtendedValidationContext, ValidationContext, Validator}
import com.xebialabs.xlrelease.domain.{PollType, ReleaseTrigger, ScheduledTrigger}
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.utils.QuartzUtils._
import org.quartz.CronExpression

class ScheduledTriggerValidator extends Validator[ScheduledTrigger] {
  override def validate(trigger: ScheduledTrigger, context: ValidationContext): Unit = {
    val extendedContext = context.asInstanceOf[ExtendedValidationContext]
    trigger.getPollType match {
      case PollType.REPEAT =>
        val errorMsg = s"Trigger has wrong periodicity '${trigger.getPeriodicity}'"
        try {
          val period = trigger.getPeriodicity.toInt
          if (period <= 0) {
            extendedContext.error(trigger, "periodicity", errorMsg)
          }
        } catch {
          case e: Exception => extendedContext.error(trigger, "periodicity", s"$errorMsg")
        }
      case PollType.CRON =>
        val cron = trigger.getPeriodicity
        val errorMsg = s"Trigger has wrong cron expression '$cron'"
        try {
          if (!CronExpression.isValidExpression(cron.asQuartzCron)) {
            extendedContext.error(trigger, "periodicity", errorMsg)
          }
        } catch {
          case e: Exception => extendedContext.error(trigger, "periodicity", s"$errorMsg")
        }
    }
  }
}

class ReleaseTriggerValidator extends Validator[ReleaseTrigger] {
  override def validate(trigger: ReleaseTrigger, context: ValidationContext): Unit = {
    val extendedContext = context.asInstanceOf[ExtendedValidationContext]
    if (isNullOrEmpty(trigger.getReleaseTitle)) {
      extendedContext.error(trigger, "releaseTitle", s"Release title cannot be blank")
    }

    validateTemplate(trigger, extendedContext)
  }

  private def validateTemplate(trigger: ReleaseTrigger, extendedContext: ExtendedValidationContext): Unit = {
    if (isNullOrEmpty(trigger.getTemplate)) {
      extendedContext.error(trigger, "template", s"Trigger does not have associated template")
    }
    if (!isNullOrEmpty(trigger.getTemplate)) {
      if (!isNullOrEmpty(trigger.getFolderId) && Ids.findFolderId(trigger.getTemplate) != trigger.getFolderId) {
        extendedContext.error(trigger, "template", s"Trigger folder id does not match template folder id")
      }
    }
  }
}
