package com.xebialabs.xlrelease.triggers.events

import com.xebialabs.xlplatform.coc.dto.SCMTraceabilityData
import com.xebialabs.xlrelease.domain.Trigger
import com.xebialabs.xlrelease.domain.events.{InterceptableAction, XLReleaseEvent}

/**
 * Common interface for domain events related to triggers.
 */
sealed trait TriggerEvent extends XLReleaseEvent

/**
 * Common interface for domain events specific to EventBasedTrigger(s).
 */
sealed trait EventBasedTriggerEvent extends TriggerEvent

sealed trait TriggerActionEvent extends InterceptableAction {
  def trigger: Trigger
}

/**
 * Event published when a release trigger is created in a template.
 *
 * @param trigger the created trigger.
 */
case class TriggerCreatedEvent(trigger: Trigger) extends TriggerEvent

/**
 * Event published when a trigger is updated.
 *
 * @param original a copy of the original trigger.
 * @param updated  the updated trigger. You can use [[original]] to compare which properties have changed.
 */
case class TriggerUpdatedEvent(original: Trigger, updated: Trigger) extends TriggerEvent

/**
 * Event published when a trigger is deleted.
 *
 * @param trigger the deleted trigger.
 */
case class TriggerDeletedEvent(trigger: Trigger) extends TriggerEvent

/**
 * Event published when a trigger is enabled.
 *
 * @param trigger enabled trigger.
 */
case class TriggerEnabledEvent(trigger: Trigger) extends TriggerEvent

/**
 * Event published when a trigger is disabled.
 *
 * @param trigger disabled trigger.
 */
case class TriggerDisabledEvent(trigger: Trigger) extends TriggerEvent

/**
 * Event published when trigger is executed successfully.
 *
 * @param trigger the executed trigger.
 */
case class TriggerExecutedEvent(trigger: Trigger, message: String, dataId: String = null) extends TriggerEvent

/**
 * Event published when trigger execution is skipped.
 *
 * @param trigger the skipped trigger.
 */
case class TriggerSkippedEvent(trigger: Trigger, message: String, dataId: String = null) extends TriggerEvent

/**
 * Event published when trigger execution failed.
 *
 * @param trigger the failed trigger.
 */
case class TriggerFailedEvent(trigger: Trigger, error: String, dataId: String = null) extends TriggerEvent

/**
 * Event published when filter is executed successfully.
 */
case class TriggerFilterPassedEvent(trigger: Trigger, dataId: String) extends EventBasedTriggerEvent

/**
 * Event published when filter execution failed.
 */
case class TriggerFilterFailedEvent(trigger: Trigger, error: String, dataId: String) extends EventBasedTriggerEvent

case class TriggerCreatingActionEvent(trigger: Trigger) extends TriggerActionEvent

case class TriggerUpdatingActionEvent(original: Trigger, trigger: Trigger) extends TriggerActionEvent

case class TriggerDeletingActionEvent(trigger: Trigger) extends TriggerActionEvent

/**
 * Event published when a trigger is created via As-code.
 *
 * @param trigger the created trigger.
 */
case class TriggerCreatedFromAsCodeEvent(trigger: Trigger, scmTraceabilityData: Option[SCMTraceabilityData]) extends TriggerEvent

/**
 * Event published when a trigger is updated via As-code.
 *
 * @param updated the updated trigger.
 */
case class TriggerUpdatedFromAsCodeEvent(updated: Trigger, scmTraceabilityData: Option[SCMTraceabilityData]) extends TriggerEvent
