package com.xebialabs.xlrelease.triggers.scheduled.quartz

import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.domain.{PollType, ScheduledTrigger}
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.triggers.scheduled.ScheduledJobService
import com.xebialabs.xlrelease.utils.QuartzUtils._
import grizzled.slf4j.Logging
import org.quartz.CronScheduleBuilder._
import org.quartz.JobBuilder._
import org.quartz.SimpleScheduleBuilder._
import org.quartz.TriggerBuilder._
import org.quartz._

import java.util.TimeZone
import scala.jdk.CollectionConverters._

class QuartzScheduledJobService(scheduler: Scheduler) extends ScheduledJobService with Logging {

  import QuartzScheduledJobService._

  override def schedule(scheduledTrigger: ScheduledTrigger): Unit = {
    logger.debug(s"Schedule trigger ${scheduledTrigger.normalizedId}")
    val triggerType = scheduledTrigger.getType.toString
    val jobClass = classOf[TriggerQuartzJob]

    val job: JobDetail = newJob(jobClass)
      .withDescription(s"Trigger '${scheduledTrigger.normalizedId}'")
      .withIdentity(scheduledTrigger.normalizedId, triggerType)
      .build()

    val quartzTrigger = newTrigger()
      .withIdentity(scheduledTrigger.normalizedId, triggerType)
      .withDescription(s"Trigger '${scheduledTrigger.normalizedId}'")
      .withSchedule(triggerSchedule(scheduledTrigger))
      .startNow()
      .build()

    if (scheduler.checkExists(quartzTrigger.getKey)) {
      scheduler.rescheduleJob(quartzTrigger.getKey, quartzTrigger)
    } else {
      scheduler.scheduleJob(job, Set(quartzTrigger).asJava, true)
    }
  }

  override def unschedule(scheduledTrigger: ScheduledTrigger): Unit = {
    logger.debug(s"Unschedule trigger ${scheduledTrigger.normalizedId}")
    val triggerKey = new TriggerKey(scheduledTrigger.normalizedId, scheduledTrigger.getType.toString)
    scheduler.unscheduleJob(triggerKey)
  }

  private def triggerSchedule(releaseTrigger: ScheduledTrigger): ScheduleBuilder[_ <: Trigger] = {
    // TODO handle misfire etc.
    val scheduleBuilder = releaseTrigger.getPollType match {
      case PollType.REPEAT =>
        simpleSchedule()
          .withIntervalInSeconds(releaseTrigger.getPeriodicity.toInt)
          .withMisfireHandlingInstructionNextWithRemainingCount()
          .repeatForever()
      case PollType.CRON =>
        val timezoneName = XlrConfig.getInstance.quartz.timezone
        val timezone: TimeZone = if (timezoneName.isEmpty) TimeZone.getDefault else TimeZone.getTimeZone(timezoneName)
        cronSchedule(releaseTrigger.getPeriodicity.asQuartzCron)
          .inTimeZone(timezone)
          .withMisfireHandlingInstructionDoNothing()
    }
    scheduleBuilder
  }
}

object QuartzScheduledJobService {

  implicit class TriggerExtension(trigger: ScheduledTrigger) {

    import com.xebialabs.xlrelease.repository.sql.persistence.CiId._

    def normalizedId: String = {
      Ids.getFolderlessId(trigger.getId.normalized)
    }
  }

}

