import React from 'react';
import { ReactWrapper } from 'enzyme';
import { mountComponentWithStore } from '../../../../../../../../../../core/xlr-ui/tests/unit/testing-utils';
import { navigation } from '../../../../../../../../../../core/xlr-ui/app/features/main-navigation/ducks/navigation.reducer';
import { ApplicationSource } from './application-source.component';
import { folderExternalDeployments, FolderExternalDeploymentsState, initialState } from '../../ducks/external-deployments.reducer';
import { mockArgoWebhookSource, mockDeployWebhookSource, mockServers, mockWebhookSources } from '../../__mocks__/external-deployments.mocks';

const { navigateTo } = navigation.actions;
const { deleteWebhookSource, getFilters, loadExternalDeployments, loadServers, loadWebhookSources, updateBreadcrumbs } = folderExternalDeployments.actions;
const dispatch = jest.fn();
describe('Application sources component', () => {
    let wrapper: ReactWrapper;
    const defaultProps = {
        folder: { id: 'folderId', title: 'Folder' },
    };
    const defaultState = {
        ...initialState,
    };
    const mountComponent = (state: FolderExternalDeploymentsState = defaultState) => {
        wrapper = mountComponentWithStore(<ApplicationSource {...defaultProps} />, dispatch, { folderExternalDeployments: state });
        expect(dispatch).toBeCalledWith(getFilters('folderId'));
        expect(dispatch).toBeCalledWith(loadWebhookSources('folderId'));
        expect(dispatch).toBeCalledWith(loadExternalDeployments());
        expect(dispatch).toBeCalledWith(updateBreadcrumbs(defaultProps.folder));
        expect(dispatch).toBeCalledWith(loadServers('folderId'));
    };

    beforeEach(() => {
        mountComponent();
    });

    afterEach(() => {
        wrapper.unmount();
    });

    const searchApplicationSources = (selector: string) => wrapper.find(`.application-source ${selector}`);

    it('should show the title and table heading with no results', () => {
        const applicationSourcesHeaders = searchApplicationSources('MuiTypographyRoot.dot-typography');
        expect(applicationSourcesHeaders.at(0).text()).toStrictEqual('Application sources');
        expect(applicationSourcesHeaders.at(1).text()).toStrictEqual('Server');
        expect(applicationSourcesHeaders.at(2).text()).toStrictEqual('Webhook');
        expect(applicationSourcesHeaders.at(3).text()).toStrictEqual('Status');
        expect(applicationSourcesHeaders.at(4).text()).toStrictEqual('Action');
        expect(searchApplicationSources('tr td p').text()).toStrictEqual('No data found');
    });

    it('should show application sources', () => {
        const disconnectedServers = mockWebhookSources[0].sourceServer ? [mockWebhookSources[0].sourceServer] : [];
        mountComponent({
            ...defaultState,
            webhookSources: mockWebhookSources,
            servers: mockServers,
            disconnectedServers,
        });
        const tableRows = searchApplicationSources('tbody tr');
        expect(tableRows.length).toStrictEqual(2);
        expect(tableRows.at(0).find('td').at(0).text()).toStrictEqual('ArgoCD');
        expect(tableRows.at(0).find('td').at(1).text()).toStrictEqual('ArgoCD');
        expect(tableRows.at(0).find('td').at(2).text()).toStrictEqual('Disconnected');
        expect(tableRows.at(0).find('td icon-edit')).toBeTruthy();
        expect(tableRows.at(0).find('td icon-delete')).toBeTruthy();
        expect(tableRows.at(1).find('td').at(0).text()).toStrictEqual('Deploy');
        expect(tableRows.at(1).find('td').at(1).text()).toStrictEqual('Deploy');
        expect(tableRows.at(1).find('td').at(2).text()).toStrictEqual('Connected');
        expect(tableRows.at(1).find('td icon-edit')).toBeTruthy();
        expect(tableRows.at(1).find('td icon-delete')).toBeTruthy();
    });

    it('should open dialog on delete click', () => {
        mountComponent({ ...defaultState, webhookSources: mockWebhookSources, servers: mockServers });
        expect(searchApplicationSources('tbody tr').at(0).find('td [title="Delete"] button').simulate('click'));

        const dialog = (selector: string) => wrapper.find(`.application-source-dialog ${selector}`);
        expect(dialog('.dot-typography')).toBeTruthy();
        expect(dialog('button.cancel-button').simulate('click'));
        expect(searchApplicationSources('tbody tr').at(0).find('td [title="Delete"] button').simulate('click'));
    });

    it('should open dialog and delete source', () => {
        mountComponent({ ...defaultState, webhookSources: mockWebhookSources, servers: mockServers });
        expect(searchApplicationSources('tbody tr').at(0).find('td [title="Delete"] button').simulate('click'));

        const dialog = (selector: string) => wrapper.find(`.application-source-dialog ${selector}`);

        expect(dialog('.dot-typography').contains('Delete application source')).toBeTruthy();
        expect(dialog('button .icon-delete').simulate('click'));
        expect(dispatch).toHaveBeenCalledWith(deleteWebhookSource(mockArgoWebhookSource.id || ''));
    });

    const mountComponentAndOpenEditMode = (row: number) => {
        const webhookSourceFilters = {
            folderFilterOptions: [],
            folderFilterValues: ['value'],
            webhookSourceId: '',
        };
        mountComponent({
            ...defaultState,
            servers: mockServers,
            webhookSourceFilters,
            webhookSources: [
                mockArgoWebhookSource,
                mockDeployWebhookSource,
                {
                    ...mockDeployWebhookSource,
                    id: null,
                    filteredApplications: undefined,
                    filteredFolders: undefined,
                    eventSource: null,
                    sourceServer: null,
                    variableMapping: undefined,
                },
            ],
        });
        expect(searchApplicationSources('tbody tr').at(row).find('td [title="Edit"] button').simulate('click'));
        const editModal = (selector: string) => wrapper.find(`.filter-applications ${selector}`);
        expect(editModal('.dot-typography').contains('Map folders')).toBeTruthy();
        return editModal;
    };

    it('should open edit mode on edit click', () => {
        mountComponentAndOpenEditMode(0);
    });

    it('should open edit mode on edit click and save', () => {
        const editModal = mountComponentAndOpenEditMode(1);
        expect(editModal('.filter-end-buttons button.primary').simulate('click'));
        expect(editModal('.dot-typography').contains('Map folders')).toBeFalsy();
    });

    it('should open edit and return to application sources with cancel', () => {
        mountComponent({ ...defaultState, webhookSources: mockWebhookSources, servers: mockServers });
        expect(searchApplicationSources('tbody tr').at(0).find('td [title="Edit"] button').simulate('click'));

        const editModal = (selector: string) => wrapper.find(`.filter-applications ${selector}`);

        expect(editModal('.filter-end-buttons button.cancel').simulate('click'));
        expect(editModal('.dot-typography').contains('Map folders')).toBeFalsy();
    });

    it('should open edit and return to application sources with back', () => {
        mountComponent({ ...defaultState, webhookSources: mockWebhookSources, servers: mockServers });
        expect(searchApplicationSources('tbody tr').at(0).find('td [title="Edit"] button').simulate('click'));

        const editModal = (selector: string) => wrapper.find(`.filter-applications ${selector}`);

        expect(editModal('button.cancel').at(0).simulate('click'));
        expect(editModal('.dot-typography').contains('Map folders')).toBeFalsy();
    });

    it('should not delete source if there is no selected source', () => {
        const webhookSourcesList = mockWebhookSources;
        webhookSourcesList[1].id = null;
        webhookSourcesList[1].sourceServer = null;
        mountComponent({ ...defaultState, webhookSources: [webhookSourcesList[1]], servers: mockServers });
        expect(searchApplicationSources('tbody tr').at(0).find('td [title="Delete"] button').simulate('click'));
        const dialog = (selector: string) => wrapper.find(`.application-source-dialog ${selector}`);

        expect(dialog('.dot-typography')).toBeTruthy();
        expect(dialog('button .icon-delete').simulate('click'));
        expect(dialog('.dot-typography').contains('Delete application source')).toBeTruthy();
    });

    it('should handle go back click', () => {
        mountComponent();
        expect(searchApplicationSources('.dot-tooltip button').simulate('click'));
        expect(dispatch).toHaveBeenCalledWith(navigateTo({ pathSuffix: `folders/${defaultProps.folder.id}/external-deployments` }));
    });
});
