import { END, EventChannel, eventChannel } from 'redux-saga';
import {
    APPLICATION_COLUMN_ID,
    APPLICATION_ORDER_BY,
    DESTINATION_COLUMN_ID,
    DESTINATION_ORDER_BY,
    ENDPOINT_TYPE,
    EVENT_TYPE,
    IMG_SRC,
    STATUS_COLUMN_ID,
    STATUS_ORDER_BY,
    TIME_COLUMN_ID,
    TIME_ORDER_BY,
    VERSION_COLUMN_ID,
    VERSION_ORDER_BY,
} from '../constants';
import { DeploymentState } from '../external-deployment.types';

export function registerEventSource(url: string): EventSource {
    return new EventSource(url);
}

export function subscribeToSse(eventSrc: EventSource): EventChannel<Event> {
    return eventChannel((emitter) => {
        eventSrc.onopen = (ev) => emitter(ev); //eslint-disable-line no-param-reassign
        eventSrc.addEventListener(EVENT_TYPE.applicationPackageCreated, (ev) => emitter(ev));
        eventSrc.addEventListener(EVENT_TYPE.applicationPackageDeleted, (ev) => emitter(ev));
        eventSrc.addEventListener(EVENT_TYPE.applicationCreated, (ev) => emitter(ev));
        eventSrc.addEventListener(EVENT_TYPE.applicationDeleted, (ev) => emitter(ev));
        eventSrc.addEventListener(EVENT_TYPE.applicationChanged, (ev) => emitter(ev));
        // eventSrc.onerror = (err) => console.error(err)

        return () => {
            eventSrc.close();
            emitter(END);
        };
    });
}

export const getDeepLink = (endpointUrl: string, applicationPath: string) => {
    return endpointUrl.replace(/\/+$/, '') + (applicationPath ? applicationPath : '');
};

export const getDiscoveryCardSubheader = (connectionType: string) => {
    switch (connectionType) {
        case ENDPOINT_TYPE.argoCd:
            return 'by Argo';
        case ENDPOINT_TYPE.deploy:
            return 'by Digital.ai';
        default:
            return '';
    }
};

export const getDiscoveryCardTitle = (connectionType: string) => {
    switch (connectionType) {
        case ENDPOINT_TYPE.argoCd:
            return 'ArgoCD';
        case ENDPOINT_TYPE.deploy:
            return 'Deploy';
        default:
            return '';
    }
};

export const getServerIcon = (serverType: string): string => {
    return (
        {
            [ENDPOINT_TYPE.deploy]: IMG_SRC.deploy,
            [ENDPOINT_TYPE.argoCd]: IMG_SRC.argoCd,
        }[serverType] || ''
    );
};

export const mapOrderByValue = (columnName: string): string => {
    switch (columnName) {
        case APPLICATION_COLUMN_ID: {
            return APPLICATION_ORDER_BY;
        }
        case STATUS_COLUMN_ID: {
            return STATUS_ORDER_BY;
        }
        case DESTINATION_COLUMN_ID: {
            return DESTINATION_ORDER_BY;
        }
        case VERSION_COLUMN_ID: {
            return VERSION_ORDER_BY;
        }
        case TIME_COLUMN_ID: {
            return TIME_ORDER_BY;
        }
        default: {
            return STATUS_ORDER_BY;
        }
    }
};

export const trimCommitHash = (hash: string): string => {
    if (hash.length > 6) return hash.substring(0, 7);
    else return hash;
};

export const getVersionLabel = (endpointType: string, label: string): string => {
    if (endpointType === ENDPOINT_TYPE.argoCd) return trimCommitHash(label);
    else return label;
};

export const getFilterCardTitle = (serverType: string) => {
    switch (serverType) {
        case ENDPOINT_TYPE.argoCd:
            return 'ArgoCD instance';
        case ENDPOINT_TYPE.deploy:
            return 'Deploy Dev instance';
        default:
            return '';
    }
};

export const generateEnvironmentTitle = (state: DeploymentState, serverType: string) => {
    switch (serverType) {
        case ENDPOINT_TYPE.argoCd:
            return state.destination + ' / ' + state.namespace;
        case ENDPOINT_TYPE.deploy:
            return state.destination;
        default:
            return '';
    }
};
