import React, { memo, ChangeEvent, useState, useEffect } from 'react';
import {
    DotColumnHeader,
    DotIcon,
    DotTable,
    DotTablePagination,
    DotThemeProvider,
    DotTooltip,
    DotTypography,
    Order,
    RowsPerPageOption,
} from '@digital-ai/dot-components';
import { ExternalDeployment as ExternalDeploymentType, ExternalDeploymentPage } from '../external-deployment.types';
import { folderExternalDeployments } from '../ducks/external-deployments.reducer';
import { Folder } from '../../../../../../../../../core/xlr-ui/app/types';
import {
    getConnectionServersSelector,
    getConditionSelector,
    getExternalDeploymentsSelector,
    getExternalDeploymentPageSelector,
    getIsLoadingSelector,
    getExternalDeploymentCountSelector,
    getConnectionErrorsSelector,
} from '../ducks/external-deployments.selectors';
import './external-deployments-table.component.less';
import { ExternalDeployment } from './external-deployment.component';
import { useAppDispatch, useAppSelector } from '../../../../../../../../../core/xlr-ui/app/js/hooks';
import { ExternalDeploymentComponentPortal } from './external-deployment-component.portal';
import { APPLICATION_COLUMN_ID, DEFAULT_PAGE, DESTINATION_COLUMN_ID, STATUS_COLUMN_ID, TIME_COLUMN_ID, VERSION_COLUMN_ID } from '../constants';
import { Search } from '../../../../../../../../../core/xlr-ui/app/react/components/search/search';

export interface ExternalDeploymentsTableProps {
    folder: Folder;
    onViewChange: () => void;
}

const { getFilters, loadExternalDeployments, setCondition, storeFilters, setPage, subscribeToSseStream, unsubscribeFromSseStream } =
    folderExternalDeployments.actions;

const ExternalDeploymentsTableComponent = ({ folder, onViewChange }: ExternalDeploymentsTableProps) => {
    const isLoading: boolean = useAppSelector(getIsLoadingSelector);
    const externalDeployments = useAppSelector(getExternalDeploymentsSelector);
    const connectionServers = useAppSelector(getConnectionServersSelector);
    const count = useAppSelector(getExternalDeploymentCountSelector);
    const page = useAppSelector(getExternalDeploymentPageSelector);
    const condition = useAppSelector(getConditionSelector);
    const connectionErrors = useAppSelector(getConnectionErrorsSelector);
    const dispatch = useAppDispatch();

    const [filterTimeout, setFilterTimeout] = useState<ReturnType<typeof setTimeout> | undefined>(undefined);

    useEffect(() => {
        dispatch(getFilters(folder.id));
        dispatch(loadExternalDeployments());
        dispatch(subscribeToSseStream());
        return () => {
            dispatch(unsubscribeFromSseStream());
        };
    }, []);

    useEffect(() => {
        dispatch(storeFilters());
    }, [page, condition]);

    const onPageChange = (page: ExternalDeploymentPage) => {
        dispatch(setPage(page));
        dispatch(loadExternalDeployments());
    };

    const onDisplayFilterChange = (event: React.ChangeEvent<HTMLInputElement>) => {
        const value = event.target.value;
        if (filterTimeout) {
            clearTimeout(filterTimeout);
            setFilterTimeout(undefined);
        }
        setFilterTimeout(
            setTimeout(() => {
                dispatch(setCondition(value));
                dispatch(loadExternalDeployments());
            }, 500),
        );
    };

    return (
        <>
            <ExternalDeploymentComponentPortal folder={folder} onViewChange={onViewChange} reSyncEnabled={true} />
            <DotThemeProvider>
                <div className="external-deployments-table">
                    <div className="external-deployments-table-title">
                        <DotTypography variant="h1">Application deployments ({count})</DotTypography>
                        <Search
                            defaultValue={condition}
                            inputPlaceholder="Type to filter"
                            onFilterChange={onDisplayFilterChange}
                            searchClassNames={['mtm', 'mbm']}
                        />
                    </div>
                    {connectionErrors.length > 0 && (
                        <DotTooltip
                            className="external-deployments-connection-errors"
                            placement="bottom-start"
                            title={
                                <div>
                                    {connectionErrors.map((e) => (
                                        <p>{e}</p>
                                    ))}
                                </div>
                            }
                        >
                            <DotTypography>
                                <DotIcon iconId="error-solid" />
                                {connectionErrors.length} disconnected servers
                            </DotTypography>
                        </DotTooltip>
                    )}
                    <DotTable
                        className="external-deployments-table-header"
                        columns={headerDefinitions}
                        data={[]}
                        loading={isLoading}
                        onUpdateData={(order: Order, orderBy: string) =>
                            onPageChange({
                                folderId: folder.id,
                                order,
                                orderBy,
                                page: page.page,
                                resultsPerPage: page.resultsPerPage,
                            })
                        }
                        order={page.order}
                        orderBy={page.orderBy}
                    />
                    {externalDeployments.map((externalDeployment: ExternalDeploymentType) => {
                        const connectionServer = connectionServers.get(externalDeployment.endpointId);
                        return (
                            connectionServer && (
                                <ExternalDeployment
                                    connectionServer={connectionServer}
                                    externalDeployment={externalDeployment}
                                    key={`${externalDeployment.applicationUid}-${externalDeployment.state.destination}`}
                                />
                            )
                        );
                    })}
                    <DotTablePagination
                        count={count}
                        onChangePage={(newPage: number) =>
                            onPageChange({
                                folderId: folder.id,
                                order: page.order,
                                orderBy: page.orderBy,
                                page: newPage,
                                resultsPerPage: page.resultsPerPage,
                            })
                        }
                        onChangeRowsPerPage={(event: ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) =>
                            onPageChange({
                                folderId: folder.id,
                                order: page.order,
                                orderBy: page.orderBy,
                                page: DEFAULT_PAGE,
                                resultsPerPage: parseInt(event.target.value) as RowsPerPageOption,
                            })
                        }
                        page={page.page}
                        rowsPerPage={page.resultsPerPage}
                        typography="body1"
                    />
                </div>
            </DotThemeProvider>
        </>
    );
};

const headerDefinitions: Array<DotColumnHeader> = [
    {
        align: 'left',
        id: 'icon',
        sortable: false,
        truncate: false,
        width: '3%',
    },
    {
        align: undefined,
        id: APPLICATION_COLUMN_ID,
        label: 'Application name',
        truncate: false,
        width: '20%',
    },
    {
        align: undefined,
        id: DESTINATION_COLUMN_ID,
        label: 'Environment',
        truncate: false,
        width: '20%',
    },
    {
        align: undefined,
        id: VERSION_COLUMN_ID,
        label: 'Version',
        truncate: false,
        width: '20%',
    },
    {
        align: undefined,
        id: STATUS_COLUMN_ID,
        label: 'Status',
        truncate: false,
        width: '16%',
    },
    {
        align: undefined,
        id: TIME_COLUMN_ID,
        label: 'Updated',
        truncate: false,
        width: '16%',
    },
    {
        align: 'right',
        id: 'deeplink',
        sortable: false,
        truncate: false,
        width: '5%',
    },
];

export const ExternalDeploymentsTable = memo(ExternalDeploymentsTableComponent);
