import React, { useEffect, useState, memo } from 'react';
import { useDispatch } from 'react-redux';
import { DotForm, DotTypography, DotButton, DotAutoComplete, AutoCompleteOption, DotCardHeader, DotCard, DotIconButton } from '@digital-ai/dot-components';
import './filter-applications-component.less';
import { Server, WebhookSource, WebhookSourceFilter } from '../../../external-deployment.types';
import { folderExternalDeployments } from '../../../ducks/external-deployments.reducer';
import { getWebhookSourceFiltersSelector } from '../../../ducks/external-deployments.selectors';
import { useAppSelector } from '../../../../../../../../../../../core/xlr-ui/app/js/hooks';
import { getFilterCardTitle, getServerIcon } from '../../../helper/utils';
import { IMG_SRC } from '../../../constants';
import { EventSourceForm } from '../event-source/application-event-source.component';

export interface FilterApplicationsProps {
    onCancel: () => void;
    onFilterBack: () => void;
    onSave: () => void;
    server: Server | undefined;
    webhookSource: WebhookSource;
}

const { loadWebhookSourceFilters, updateWebhookSourceFilters } = folderExternalDeployments.actions;

const FilterApplicationsComponent = (props: FilterApplicationsProps) => {
    const { webhookSource, onFilterBack, onCancel, onSave, server } = props;

    const webhookSourceFilters: WebhookSourceFilter | undefined = useAppSelector(getWebhookSourceFiltersSelector);

    const [isWebhookSourceForm, setIsWebhookSourceForm] = useState<boolean>(false);
    const [folderOptions, setFolderOptions] = useState<Array<string>>([]);
    const [selectedFolders, setSelectedFolders] = useState<Array<string>>([]);

    const dispatch = useDispatch();

    const saveApplicationFilters = () => {
        dispatch(
            updateWebhookSourceFilters({
                webhookSourceId: webhookSource.id!,
                filteredFolders: selectedFolders,
            }),
        );
        onSave();
    };

    const handleSelectedFoldersChange = (v: Array<AutoCompleteOption>) => {
        setSelectedFolders(v.map((value) => value.title));
    };

    const onEventSourceFormClose = () => {
        setIsWebhookSourceForm(false);
    };

    useEffect(() => {
        dispatch(loadWebhookSourceFilters(webhookSource.id!));
    }, []);

    useEffect(() => {
        setFolderOptions(webhookSourceFilters ? webhookSourceFilters.folderFilterOptions : []);
        setSelectedFolders(webhookSourceFilters ? webhookSourceFilters.folderFilterValues : []);
    }, [webhookSourceFilters]);

    const generateApplicationFilters = () => {
        return (
            <div className="filter-applications">
                <DotForm onSubmit={saveApplicationFilters}>
                    <DotTypography variant="h1">Map folders</DotTypography>
                    <DotAutoComplete
                        className="folders-filter"
                        freesolo={false}
                        helperText={'Map folders to display applications from'}
                        inputId="webhook-source-folders-filter"
                        label="Choose folders"
                        onChange={(e, v) => handleSelectedFoldersChange(v as Array<AutoCompleteOption>)}
                        options={folderOptions.map((option) => ({ id: option, title: option }))}
                        value={selectedFolders.map((option) => ({ id: option, title: option }))}
                    />
                    <DotCard className="filter-card">
                        <DotCardHeader
                            avatar={<img alt={server?.type} src={getServerIcon(server ? server.type! : '')} />}
                            className="card-header-inline"
                            subheader={server?.url}
                            subheaderSize="small"
                            title={getFilterCardTitle(server ? server.type! : '')}
                            titleSize="small"
                        />
                    </DotCard>
                    <DotCard className="filter-card">
                        <DotCardHeader
                            action={
                                <DotIconButton
                                    color="inherit"
                                    data-testid="filter-webhook-edit-action"
                                    iconId="edit"
                                    iconSize="small"
                                    onClick={() => setIsWebhookSourceForm(true)}
                                    size="medium"
                                    tooltip="Edit webhook endpoint"
                                />
                            }
                            avatar={<img alt={webhookSource.title} src={IMG_SRC.webhookSource} />}
                            className="card-header-inline"
                            title={webhookSource.title}
                            titleSize="small"
                        />
                    </DotCard>
                    <div>
                        <DotButton className="button cancel" onClick={onFilterBack} type="text">
                            Back
                        </DotButton>
                        <span className="filter-end-buttons">
                            <DotButton className="button cancel" onClick={onCancel} type="outlined">
                                Cancel
                            </DotButton>
                            <DotButton className="button primary" disabled={selectedFolders.length === 0} isSubmit={true}>
                                Done
                            </DotButton>
                        </span>
                    </div>
                </DotForm>
            </div>
        );
    };

    const generateWebhookSourceForm = () => {
        return <EventSourceForm closeForm={onEventSourceFormClose} eventSourceId={webhookSource.eventSource!} />;
    };

    return (
        <>
            {!isWebhookSourceForm && generateApplicationFilters()}
            {isWebhookSourceForm && generateWebhookSourceForm()}
        </>
    );
};

export const FilterApplications = memo(FilterApplicationsComponent);
