package com.xebialabs.xlrelease.status.sse.service

import com.fasterxml.jackson.databind.ObjectMapper
import com.fasterxml.jackson.datatype.joda.JodaModule
import com.fasterxml.jackson.module.scala.DefaultScalaModule
import com.xebialabs.xlrelease.status.webhook.events._
import grizzled.slf4j.Logging
import org.springframework.http.MediaType
import org.springframework.stereotype.Service
import org.springframework.web.servlet.mvc.method.annotation.SseEmitter
import org.springframework.web.servlet.mvc.method.annotation.SseEmitter.SseEventBuilder

import java.io.EOFException
import java.util.ConcurrentModificationException
import java.util.concurrent.{CompletableFuture, CopyOnWriteArrayList}

trait ServerSentEventsService {
  def add(): SseEmitter

  def send(id: String, event: ExternalDeploymentEvent): Unit
}

@Service
class ServerSentEventsServiceImpl extends ServerSentEventsService with Logging {

  private val emitters = new CopyOnWriteArrayList[SseEmitter]()
  private val mapper = new ObjectMapper()
  mapper.registerModule(DefaultScalaModule)
  mapper.registerModule(new JodaModule)

  override def add(): SseEmitter = {
    val emitter = new SseEmitter(Long.MaxValue)
    emitters.add(emitter)

    emitter.onCompletion(() => {
      if (emitters.remove(emitter)) logger.info(s"Emitter $emitter completed")
    })

    emitter.onTimeout(() => {
      if (emitters.remove(emitter)) {
        logger.info(s"Emitter $emitter timed out")
        emitter.complete()
      }
    })

    emitter.onError(e => {
      if (emitters.remove(emitter)) {
        logger.info(s"An error occurred in emitter $emitter: $e")
        emitter.complete()
      }
    })

    logger.info(s"New emitter $emitter has been registered")
    emitter
  }

  def send(id: String, event: ExternalDeploymentEvent): Unit = {
    val sseEvent = buildSseEvent(EndpointExternalDeploymentEvent(id, event))

    emitters.forEach(emitter => {
      CompletableFuture.runAsync(() => {
        emitter.send(sseEvent)
      }).whenComplete((_, ex) => {
        Option(ex.getCause).map {
          case _: EOFException | _: ConcurrentModificationException =>
            if (emitters.remove(emitter)) {
              emitter.complete()
              logger.info(s"Client connection for emitter $emitter was terminated, removing emitter $emitter")
            }
          case ex: Throwable =>
            if (emitters.remove(emitter)) {
              emitter.completeWithError(ex)
              logger.error(s"Emitter $emitter failed: $ex")
            }
          case _ => // did not happen
        }
      })
    })
  }

  private def buildSseEvent(event: EndpointExternalDeploymentEvent): SseEventBuilder = {
    val sseEvent: SseEventBuilder = SseEmitter.event()
      .data(mapper.writeValueAsString(event), MediaType.TEXT_PLAIN)

    event.state match {
      case e: CreateStatusEvent if isApplicationEvent(e) => sseEvent.name("application-created")
      case e: DeleteStatusEvent if isApplicationEvent(e) => sseEvent.name("application-deleted")
      case _: CreateStatusEvent => sseEvent.name("application-package-created")
      case _: DeleteStatusEvent => sseEvent.name("application-package-deleted")
      case _: UpdateStatusEvent => sseEvent.name("application-changed")
      case _ => // ignore
    }

    sseEvent
  }

  private def isApplicationEvent(event: BaseExternalDeploymentEvent) = {
    Option(event.destination).isDefined || Option(event.namespace).isDefined
  }
}

sealed trait ServerSentEvent {
  val endpointId: String
}

case class EndpointExternalDeploymentEvent(override val endpointId: String, state: ExternalDeploymentEvent) extends ServerSentEvent
