import { all, call, CallEffect, put, select, take, takeEvery } from 'redux-saga/effects';
import angular from 'angular';
import { AxiosError } from 'axios';
import { PayloadAction } from '@reduxjs/toolkit';
import { EventChannel } from 'redux-saga';
import { folderExternalDeployments, FolderExternalDeploymentsState } from './external-deployments.reducer';
import { DeleteStatusEvent, EndpointExternalDeploymentEvent, ExternalDeployment, UpdateStatusEvent } from '../external-deployment.types';
import { CustomConfiguration } from '../../../../../../../../../core/xlr-ui/app/types';
import { mapOrderByValue, registerEventSource, subscribeToSse } from '../helper/utils';
import { DEFAULT_MAX_AGE, EVENT_TYPE, MAX_AGE_NO_CACHE, SSE_REGISTRATION_URL } from '../constants';
import { getApplicationStatusChannelSelector, getFolderExternalDeployments } from './external-deployments.selectors';
import ToastrFactory from '../../../../../../../../../core/xlr-ui/app/js/util/toastrFactory';
import { httpGET } from '../../../../../../../../../core/xlr-ui/app/features/common/services/http';

const {
    createApplicationStatus,
    deleteApplicationStatus,
    loadExternalDeployments,
    setIsLoading,
    setConnectionServers,
    setExternalDeployments,
    setExternalDeploymentsCount,
    setMaxAge,
    setSseStreamChannel,
    subscribeToSseStream,
    unsubscribeFromSseStream,
    updateApplicationStatus,
} = folderExternalDeployments.actions;
export const toaster = ToastrFactory();

export function* executeCreateApplicationPackageStatusAction() {
    yield put(setMaxAge(MAX_AGE_NO_CACHE));
    yield call(executeLoadExternalDeploymentsAction);
    yield put(setMaxAge(DEFAULT_MAX_AGE));
}

export function* executeUpdateApplicationPackageStatusAction(action: PayloadAction<EndpointExternalDeploymentEvent<UpdateStatusEvent>>) {
    const { connectionServers, externalDeployments }: FolderExternalDeploymentsState = yield select(getFolderExternalDeployments);
    const { endpointId, state: event } = action.payload;
    let found = false;
    const updatedDeployments: Array<ExternalDeployment> = externalDeployments.map((deployment): ExternalDeployment => {
        const server = connectionServers.get(deployment.endpointId);
        if (deployment.applicationUid === event.applicationUid && event.destination === deployment.state.destination && server?.id === endpointId) {
            found = true;
            return {
                ...deployment,
                ...event,
            };
        }
        return deployment;
    });
    if (found) {
        yield put(setExternalDeployments(updatedDeployments));
    } else {
        yield call(executeCreateApplicationPackageStatusAction);
    }
}

export function* executeDeleteApplicationPackageStatusAction(action: PayloadAction<EndpointExternalDeploymentEvent<DeleteStatusEvent>>) {
    const { connectionServers, externalDeployments }: FolderExternalDeploymentsState = yield select(getFolderExternalDeployments);
    const deleteEvent: DeleteStatusEvent = action.payload.state;
    const eventEndpointId = action.payload.endpointId;
    const updatedDeployments = externalDeployments.filter(({ applicationUid, endpointId, state }) => {
        const server = connectionServers.get(endpointId);
        return applicationUid !== deleteEvent.applicationUid || server?.id !== eventEndpointId || state.destination !== deleteEvent.destination;
    });
    yield put(setExternalDeployments(updatedDeployments));
}

export function* withLoadingState(effect: CallEffect) {
    try {
        yield put(setIsLoading(true));
        yield effect;
    } finally {
        yield put(setIsLoading(false));
    }
}

export function* executeLoadExternalDeploymentsAction() {
    yield call(withLoadingState, call(executeLoadExternalDeployments));
}

export function* executeLoadExternalDeployments() {
    const { page, maxAge, condition }: FolderExternalDeploymentsState = yield select(getFolderExternalDeployments);
    if (!page || !page.folderId) {
        return;
    }
    try {
        const {
            data: { count, connectionServers, externalDeployments },
        } = yield call(
            httpGET,
            `/external-deployment?max-age=${maxAge}&folderId=${page.folderId}&page=${page.page}&resultsPerPage=${page.resultsPerPage}&orderBy=${mapOrderByValue(
                page.orderBy,
            )}&order=${page.order.toUpperCase()}&condition=${condition}`,
            true,
        );
        yield put(setConnectionServers(new Map(Object.entries(connectionServers))));
        yield put(setExternalDeployments(externalDeployments));
        yield put(setExternalDeploymentsCount(count));
    } catch (e: unknown) {
        const err = e as AxiosError;
        const errServerData: CustomConfiguration = err.response?.data;
        const errorMessage =
            'Error fetching status data. Check connection to ' + errServerData.title + ' HTTP Connection or check application logs for more details.';
        yield call(toaster.error, errorMessage);
    }
}

export function* executeSseStreamSubscribeAction() {
    const existingChannel: EventChannel<Event> = yield select(getApplicationStatusChannelSelector);
    if (existingChannel) {
        yield call(existingChannel.close);
    }
    const eventSrc: EventSource = yield call(registerEventSource, SSE_REGISTRATION_URL);
    const channel: EventChannel<Event> = yield call(subscribeToSse, eventSrc);
    yield put(setSseStreamChannel(channel));

    while (true) {
        const event: MessageEvent = yield take(channel);
        const data = angular.fromJson(event.data);
        switch (event.type) {
            case EVENT_TYPE.applicationPackageCreated:
                // might be needed in the future
                break;
            case EVENT_TYPE.applicationPackageDeleted:
                // might be needed in the future
                break;
            case EVENT_TYPE.applicationCreated:
                yield put(createApplicationStatus(data));
                break;
            case EVENT_TYPE.applicationDeleted:
                yield put(deleteApplicationStatus(data));
                break;
            case EVENT_TYPE.applicationChanged:
                yield put(updateApplicationStatus(data));
                break;
            default:
            //do nothing
        }
    }
}

export function* executeSseStreamUnsubscribeAction() {
    const channel: EventChannel<Event> = yield select(getApplicationStatusChannelSelector);
    if (channel) {
        yield call(channel.close);
    }
    yield put(setSseStreamChannel(undefined));
}

export default function* externalDeploymentsRootSaga() {
    yield all([
        takeEvery(loadExternalDeployments, executeLoadExternalDeploymentsAction),
        takeEvery(createApplicationStatus, executeCreateApplicationPackageStatusAction),
        takeEvery(updateApplicationStatus, executeUpdateApplicationPackageStatusAction),
        takeEvery(deleteApplicationStatus, executeDeleteApplicationPackageStatusAction),
        takeEvery(subscribeToSseStream, executeSseStreamSubscribeAction),
        takeEvery(unsubscribeFromSseStream, executeSseStreamUnsubscribeAction),
    ]);
}
