import React, { memo, useEffect, ChangeEvent, useState } from 'react';
import { DotColumnHeader, DotTable, DotTablePagination, DotThemeProvider, DotTypography, Order, RowsPerPageOption } from '@digital-ai/dot-components';
import { ExternalDeployment as ExternalDeploymentType } from '../external-deployment.types';
import { folderExternalDeployments } from '../ducks/external-deployments.reducer';
import { Folder } from '../../../../../../../../../core/xlr-ui/app/types';
import {
    getConnectionServersSelector,
    getExternalDeploymentsSelector,
    getExternalDeploymentPageSelector,
    getIsLoadingSelector,
    getExternalDeploymentCountSelector,
} from '../ducks/external-deployments.selectors';
import './external-deployments-table.component.less';
import { ExternalDeployment } from './external-deployment.component';
import { useAppDispatch, useAppSelector } from '../../../../../../../../../core/xlr-ui/app/js/hooks';
import {
    APPLICATION_COLUMN_ID,
    DEFAULT_ORDER,
    DEFAULT_ORDER_BY,
    DEFAULT_PAGE,
    DEFAULT_RESULTS_PER_PAGE,
    DESTINATION_COLUMN_ID,
    STATUS_COLUMN_ID,
    TIME_COLUMN_ID,
    VERSION_COLUMN_ID,
    DEFAULT_CONDITION,
} from '../constants';
import { Search } from '../../../../../../../../../core/xlr-ui/app/react/components/search/search';

export interface ExternalDeploymentsTableProps {
    folder: Folder;
}

const { loadExternalDeployments, subscribeToSseStream, unsubscribeFromSseStream, setCondition } = folderExternalDeployments.actions;

const ExternalDeploymentsTableComponent = ({ folder }: ExternalDeploymentsTableProps) => {
    const isLoading: boolean = useAppSelector(getIsLoadingSelector);
    const externalDeployments = useAppSelector(getExternalDeploymentsSelector);
    const connectionServers = useAppSelector(getConnectionServersSelector);
    const count = useAppSelector(getExternalDeploymentCountSelector);
    const page = useAppSelector(getExternalDeploymentPageSelector);
    const dispatch = useAppDispatch();

    const [filterTimeout, setFilterTimeout] = useState<ReturnType<typeof setTimeout> | undefined>(undefined);

    useEffect(() => {
        dispatch(setCondition(DEFAULT_CONDITION));
        dispatch(
            loadExternalDeployments({
                folderId: folder.id,
                order: DEFAULT_ORDER,
                orderBy: DEFAULT_ORDER_BY,
                page: DEFAULT_PAGE,
                resultsPerPage: DEFAULT_RESULTS_PER_PAGE,
            }),
        );
        dispatch(subscribeToSseStream());
        return () => {
            dispatch(unsubscribeFromSseStream());
        };
    }, []);

    const onDisplayFilterChange = (event: React.ChangeEvent<HTMLInputElement>) => {
        const value = event.target.value;
        if (filterTimeout) {
            clearTimeout(filterTimeout);
            setFilterTimeout(undefined);
        }
        setFilterTimeout(
            setTimeout(() => {
                dispatch(setCondition(value));
                dispatch(
                    loadExternalDeployments({
                        folderId: folder.id,
                        order: page.order,
                        orderBy: page.orderBy,
                        page: page.page,
                        resultsPerPage: page.resultsPerPage,
                    }),
                );
            }, 500),
        );
    };

    return (
        <DotThemeProvider>
            <div className="external-deployments-table">
                <div className="external-deployments-table-title">
                    <DotTypography variant="h1">Application deployments ({externalDeployments.length})</DotTypography>
                    <Search inputPlaceholder="Type to filter" onFilterChange={onDisplayFilterChange} searchClassNames={['mtm', 'mbm']} />
                </div>
                <DotTable
                    className="external-deployments-table-header"
                    columns={headerDefinitions}
                    data={[]}
                    loading={isLoading}
                    onUpdateData={(order: Order, orderBy: string) =>
                        dispatch(
                            loadExternalDeployments({
                                folderId: folder.id,
                                order,
                                orderBy,
                                page: page.page,
                                resultsPerPage: page.resultsPerPage,
                            }),
                        )
                    }
                    order={page.order}
                    orderBy={page.orderBy}
                />
                {externalDeployments.map((externalDeployment: ExternalDeploymentType) => {
                    const connectionServer = connectionServers.get(externalDeployment.endpointId);
                    return (
                        connectionServer && (
                            <ExternalDeployment
                                connectionServer={connectionServer}
                                externalDeployment={externalDeployment}
                                key={`${externalDeployment.applicationUid}-${externalDeployment.state.destination}`}
                            />
                        )
                    );
                })}
                <DotTablePagination
                    count={count}
                    onChangePage={(newPage: number) =>
                        dispatch(
                            loadExternalDeployments({
                                folderId: folder.id,
                                order: page.order,
                                orderBy: page.orderBy,
                                page: newPage,
                                resultsPerPage: page.resultsPerPage,
                            }),
                        )
                    }
                    onChangeRowsPerPage={(event: ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) =>
                        dispatch(
                            loadExternalDeployments({
                                folderId: folder.id,
                                order: page.order,
                                orderBy: page.orderBy,
                                page: DEFAULT_PAGE,
                                resultsPerPage: parseInt(event.target.value) as RowsPerPageOption,
                            }),
                        )
                    }
                    page={page.page}
                    rowsPerPage={page.resultsPerPage}
                    typography="body1"
                />
            </div>
        </DotThemeProvider>
    );
};

const headerDefinitions: Array<DotColumnHeader> = [
    {
        align: 'left',
        id: 'icon',
        sortable: false,
        truncate: false,
        width: '3%',
    },
    {
        align: undefined,
        id: APPLICATION_COLUMN_ID,
        label: 'Application name',
        truncate: false,
        width: '20%',
    },
    {
        align: undefined,
        id: DESTINATION_COLUMN_ID,
        label: 'Environment',
        truncate: false,
        width: '20%',
    },
    {
        align: undefined,
        id: VERSION_COLUMN_ID,
        label: 'Version',
        truncate: false,
        width: '20%',
    },
    {
        align: undefined,
        id: STATUS_COLUMN_ID,
        label: 'Status',
        truncate: false,
        width: '16%',
    },
    {
        align: undefined,
        id: TIME_COLUMN_ID,
        label: 'Updated',
        truncate: false,
        width: '16%',
    },
    {
        align: 'right',
        id: 'deeplink',
        sortable: false,
        truncate: false,
        width: '5%',
    },
];

export const ExternalDeploymentsTable = memo(ExternalDeploymentsTableComponent);
