import { SagaIterator } from 'redux-saga';
import { testSaga } from 'redux-saga-test-plan';
import angular from 'angular';
import constant from 'lodash/constant';
import { all, call, put, select, takeEvery } from 'redux-saga/effects';
// eslint-disable-next-line @typescript-eslint/ban-ts-comment
// @ts-ignore
import { EventSource } from 'mocksse';
import externalDeploymentsRootSaga, {
    executeCreateApplicationPackageStatusAction,
    executeDeleteApplicationPackageStatusAction,
    executeLoadExternalDeployments,
    executeLoadExternalDeploymentsAction,
    executeSseStreamSubscribeAction,
    executeSseStreamUnsubscribeAction,
    executeUpdateApplicationPackageStatusAction,
    toaster,
    withLoadingState,
} from './external-deployments.saga';
import { folderExternalDeployments } from './external-deployments.reducer';
import { CreateStatusEvent, DeleteStatusEvent, EndpointExternalDeploymentEvent, ExternalDeployment, UpdateStatusEvent } from '../external-deployment.types';
import { getApplicationStatusChannelSelector, getFolderExternalDeployments } from './external-deployments.selectors';
import { CustomConfiguration } from '../../../../../../../../../core/xlr-ui/app/types';
import { DEFAULT_MAX_AGE, EVENT_TYPE, MAX_AGE_NO_CACHE, SSE_REGISTRATION_URL } from '../constants';
import { registerEventSource, subscribeToSse } from '../helper/utils';
import { httpGET } from '../../../../../../../../../core/xlr-ui/app/features/common/services/http';
import {
    baseExternalDeploymentEventMock,
    connectionServerMock,
    connectionServersMock,
    deploymentStateMock,
    deploymentStateWithoutDateTimeMock,
    externalDeploymentMock,
} from '../__mocks__/external-deployments.mocks';

const {
    createApplicationStatus,
    deleteApplicationStatus,
    loadExternalDeployments,
    setConnectionServers,
    setExternalDeployments,
    setIsLoading,
    setMaxAge,
    setSseStreamChannel,
    subscribeToSseStream,
    unsubscribeFromSseStream,
    updateApplicationStatus,
} = folderExternalDeployments.actions;

describe('externalDeploymentsSaga', () => {
    it('should yield all effects', () => {
        const gen: SagaIterator = externalDeploymentsRootSaga();
        expect(gen.next().value).toStrictEqual(
            all([
                takeEvery(loadExternalDeployments, executeLoadExternalDeploymentsAction),
                takeEvery(createApplicationStatus, executeCreateApplicationPackageStatusAction),
                takeEvery(updateApplicationStatus, executeUpdateApplicationPackageStatusAction),
                takeEvery(deleteApplicationStatus, executeDeleteApplicationPackageStatusAction),
                takeEvery(subscribeToSseStream, executeSseStreamSubscribeAction),
                takeEvery(unsubscribeFromSseStream, executeSseStreamUnsubscribeAction),
            ]),
        );
        expect(gen.next().done).toBe(true);
    });

    describe('executeCreateApplicationPackageStatusAction', () => {
        it('should call executeLoadExternalDeploymentsAction', () => {
            const gen: SagaIterator = executeCreateApplicationPackageStatusAction();
            expect(gen.next().value).toStrictEqual(put(setMaxAge(MAX_AGE_NO_CACHE)));
            expect(gen.next().value).toStrictEqual(call(executeLoadExternalDeploymentsAction));
            expect(gen.next().value).toStrictEqual(put(setMaxAge(DEFAULT_MAX_AGE)));
            expect(gen.next().done).toBeTruthy();
        });
    });

    const externalDeployments: Array<ExternalDeployment> = [externalDeploymentMock];

    const createStatusEvent: EndpointExternalDeploymentEvent<CreateStatusEvent> = {
        endpointId: 'Configuration/Xl Deploy',
        state: {
            ...baseExternalDeploymentEventMock,
            statusGroup: 'success',
            versionTag: { label: '1.0' },
        },
    };

    const deleteStatusEvent: EndpointExternalDeploymentEvent<DeleteStatusEvent> = {
        endpointId: 'Configuration/Xl Deploy',
        state: {
            ...baseExternalDeploymentEventMock,
            versionTag: { label: '1.0' },
        },
    };

    const updateStatusEvent: EndpointExternalDeploymentEvent<UpdateStatusEvent> = {
        endpointId: 'Configuration/Xl Deploy',
        state: {
            ...baseExternalDeploymentEventMock,
            state: deploymentStateWithoutDateTimeMock,
            statusGroup: 'in-progress',
        },
    };

    const connectionServersObject = { '12': connectionServerMock };
    const connectionServers = connectionServersMock;

    describe('executeUpdateApplicationPackageStatusAction', () => {
        it('should update existing entry', () => {
            const gen: SagaIterator = executeUpdateApplicationPackageStatusAction({
                payload: updateStatusEvent,
                type: 'whatever',
            });
            const updatedDeployments = [
                {
                    ...externalDeployments[0],
                    ...updateStatusEvent.state,
                },
            ];
            expect(gen.next().value).toStrictEqual(select(getFolderExternalDeployments));
            expect(gen.next({ connectionServers, externalDeployments }).value).toStrictEqual(put(setExternalDeployments(updatedDeployments)));
            expect(gen.next().done).toBeTruthy();
        });

        it('should reload the deployments', () => {
            const updateStatusEventNewApp: EndpointExternalDeploymentEvent<UpdateStatusEvent> = {
                endpointId: 'Configuration/Xl Deploy',
                state: {
                    ...baseExternalDeploymentEventMock,
                    applicationUid: 'different uid',
                    state: deploymentStateMock,
                    statusGroup: 'in-progress',
                },
            };
            const gen: SagaIterator = executeUpdateApplicationPackageStatusAction({
                payload: updateStatusEventNewApp,
                type: 'whatever',
            });
            expect(gen.next().value).toStrictEqual(select(getFolderExternalDeployments));
            expect(gen.next({ connectionServers, externalDeployments }).value).toStrictEqual(call(executeCreateApplicationPackageStatusAction));
            expect(gen.next().done).toBeTruthy();
        });
    });

    describe('executeDeleteApplicationPackageStatusAction', () => {
        it('should remove existing deployment', () => {
            const gen: SagaIterator = executeDeleteApplicationPackageStatusAction({
                payload: deleteStatusEvent,
                type: 'whatever',
            });
            expect(gen.next().value).toStrictEqual(select(getFolderExternalDeployments));
            expect(gen.next({ connectionServers, externalDeployments }).value).toStrictEqual(put(setExternalDeployments([])));
            expect(gen.next().done).toBeTruthy();
        });
    });

    describe('withLoadingState', () => {
        it('should set and reset isLoading flag', () => {
            const alwaysFalse = constant(false);
            const gen: SagaIterator = withLoadingState(call(alwaysFalse));
            expect(gen.next().value).toStrictEqual(put(setIsLoading(true)));
            expect(gen.next().value).toStrictEqual(call(alwaysFalse));
            expect(gen.next().value).toStrictEqual(put(setIsLoading(false)));
            expect(gen.next().done).toBeTruthy();
        });
    });

    describe('executeLoadExternalDeploymentsAction', () => {
        it('should call executeLoadExternalDeployments with loading', () => {
            const gen: SagaIterator = executeLoadExternalDeploymentsAction();
            expect(gen.next().value).toStrictEqual(call(withLoadingState, call(executeLoadExternalDeployments)));
            expect(gen.next().done).toBeTruthy();
        });
    });

    describe('executeLoadExternalDeployments', () => {
        const folderId = 'folderId';
        const maxAge = 10000;

        it('should not fetch anything', () => {
            const gen: SagaIterator = executeLoadExternalDeployments();
            expect(gen.next().value).toStrictEqual(select(getFolderExternalDeployments));
            expect(gen.next({ maxAge }).done).toBeTruthy();
        });

        it('should show toastr', () => {
            const gen: SagaIterator = executeLoadExternalDeployments();
            const conf: CustomConfiguration = {
                id: 'serverId',
                title: 'Deploy Server',
                type: 'xldeploy.XLDeployServer',
            };
            const errorMessage = 'Error fetching status data. Check connection to Deploy Server HTTP Connection or check application logs for more details.';

            expect(gen.next().value).toStrictEqual(select(getFolderExternalDeployments));
            expect(gen.next({ folderId, maxAge }).value).toStrictEqual(call(httpGET, `/external-deployment?max-age=${maxAge}&folderId=${folderId}`, true));
            expect(gen.throw?.({ response: { data: conf } }).value).toStrictEqual(call(toaster.error, errorMessage));
            expect(gen.next().done).toBeTruthy();
        });

        it('should fetch deployments and connections', () => {
            const gen: SagaIterator = executeLoadExternalDeployments();
            expect(gen.next().value).toStrictEqual(select(getFolderExternalDeployments));
            expect(gen.next({ folderId, maxAge }).value).toStrictEqual(call(httpGET, `/external-deployment?max-age=${maxAge}&folderId=${folderId}`, true));
            expect(gen.next({ data: { connectionServers: connectionServersObject, externalDeployments } }).value).toStrictEqual(
                put(setConnectionServers(new Map(Object.entries(connectionServersObject)))),
            );
            expect(gen.next({ data: { connectionServers: connectionServersObject, externalDeployments } }).value).toStrictEqual(
                put(setExternalDeployments(externalDeployments)),
            );
            expect(gen.next().done).toBeTruthy();
        });
    });

    const mockEventSource = new EventSource(SSE_REGISTRATION_URL);
    const mockEventChannel = subscribeToSse(mockEventSource);

    describe('executeSseStreamSubscribeAction', () => {
        it('should register event source and listen to events', () => {
            const saga = testSaga(executeSseStreamSubscribeAction);
            saga.next().select(getApplicationStatusChannelSelector);
            saga.next().call(registerEventSource, SSE_REGISTRATION_URL);
            saga.next(mockEventSource).call(subscribeToSse, mockEventSource);
            saga.next(mockEventChannel).put(setSseStreamChannel(mockEventChannel));
            saga.next().take(mockEventChannel);
            saga.next({ data: angular.toJson({}), type: EVENT_TYPE.applicationPackageCreated }).take(mockEventChannel);
            saga.next({ data: angular.toJson({}), type: EVENT_TYPE.applicationPackageDeleted }).take(mockEventChannel);
            saga.next({ data: angular.toJson(createStatusEvent), type: EVENT_TYPE.applicationCreated }).put(createApplicationStatus(createStatusEvent));
            saga.next().take(mockEventChannel);
            saga.next({ data: angular.toJson(deleteStatusEvent), type: EVENT_TYPE.applicationDeleted }).put(deleteApplicationStatus(deleteStatusEvent));
            saga.next().take(mockEventChannel);
            saga.next({ data: angular.toJson(updateStatusEvent), type: EVENT_TYPE.applicationChanged }).put(updateApplicationStatus(updateStatusEvent));
            saga.next().take(mockEventChannel);
            saga.next({ data: angular.toJson({}), type: 'unknown' }).finish();
            saga.next().isDone();
        });

        it('should close the existing channel', () => {
            const saga = testSaga(executeSseStreamSubscribeAction);
            saga.next().select(getApplicationStatusChannelSelector);
            saga.next(mockEventChannel).call(mockEventChannel.close);
            saga.next(mockEventChannel).call(registerEventSource, SSE_REGISTRATION_URL);
            saga.next(mockEventSource).call(subscribeToSse, mockEventSource);
            saga.next(mockEventChannel).put(setSseStreamChannel(mockEventChannel)).finish();
            saga.next().isDone();
        });
    });

    describe('executeSseStreamUnsubscribeAction', () => {
        it('should close and unset sse channel', () => {
            const gen: SagaIterator = executeSseStreamUnsubscribeAction();
            expect(gen.next().value).toStrictEqual(select(getApplicationStatusChannelSelector));
            expect(gen.next(mockEventChannel).value).toStrictEqual(call(mockEventChannel.close));
            expect(gen.next().value).toStrictEqual(put(setSseStreamChannel(undefined)));
            expect(gen.next().done).toBeTruthy();
        });

        it('should unset sse channel', () => {
            const gen: SagaIterator = executeSseStreamUnsubscribeAction();
            expect(gen.next().value).toStrictEqual(select(getApplicationStatusChannelSelector));
            expect(gen.next().value).toStrictEqual(put(setSseStreamChannel(undefined)));
            expect(gen.next().done).toBeTruthy();
        });
    });
});
