import React, { memo, useEffect } from 'react';
import { DotColumnHeader, DotTable, DotThemeProvider, DotTypography } from '@digital-ai/dot-components';
import { ExternalDeployment as ExternalDeploymentType } from '../external-deployment.types';
import { folderExternalDeployments } from '../ducks/external-deployments.reducer';
import { Folder } from '../../../../../../../../../core/xlr-ui/app/types';
import { getConnectionServersSelector, getExternalDeploymentsSelector, getIsLoadingSelector } from '../ducks/external-deployments.selectors';
import './external-deployments-table.component.less';
import { ExternalDeployment } from './external-deployment.component';
import { useAppDispatch, useAppSelector } from '../../../../../../../../../core/xlr-ui/app/js/hooks';

export interface ExternalDeploymentsTableProps {
    folder: Folder;
}

const { loadExternalDeployments, subscribeToSseStream, unsubscribeFromSseStream } = folderExternalDeployments.actions;

const ExternalDeploymentsTableComponent = ({ folder }: ExternalDeploymentsTableProps) => {
    const isLoading: boolean = useAppSelector(getIsLoadingSelector);
    const externalDeployments = useAppSelector(getExternalDeploymentsSelector);
    const connectionServers = useAppSelector(getConnectionServersSelector);
    const dispatch = useAppDispatch();

    useEffect(() => {
        dispatch(loadExternalDeployments(folder.id));
        dispatch(subscribeToSseStream());
        return () => {
            dispatch(unsubscribeFromSseStream());
        };
    }, []);

    return (
        <DotThemeProvider>
            <div className="external-deployments-table">
                <DotTypography variant="h1">Application deployments ({externalDeployments.length})</DotTypography>
                <DotTable className="external-deployments-table-header" columns={headerDefinitions} data={[]} />
                {externalDeployments.map((externalDeployment: ExternalDeploymentType) => {
                    const connectionServer = connectionServers.get(externalDeployment.endpointId);
                    return (
                        connectionServer && (
                            <ExternalDeployment
                                connectionServer={connectionServer}
                                externalDeployment={externalDeployment}
                                key={`${externalDeployment.applicationUid}-${externalDeployment.state.destination}`}
                            />
                        )
                    );
                })}
                {!isLoading && externalDeployments.length === 0 && <div className="alert notice fc-list-loaded">No external deployments found.</div>}
                {isLoading && (
                    <div className="spinner">
                        <img alt="Loading..." src="static/@project.version@/styles/img/big-ajax-loader.gif" />
                    </div>
                )}
            </div>
        </DotThemeProvider>
    );
};

const headerDefinitions: Array<DotColumnHeader> = [
    {
        align: 'left',
        id: 'icon',
        sortable: false,
        truncate: false,
        width: '3%',
    },
    {
        align: undefined,
        id: 'applicationName',
        label: 'Application name',
        sortable: false,
        truncate: false,
        width: '20%',
    },
    {
        align: undefined,
        id: 'environment',
        label: 'Environment',
        sortable: false,
        truncate: false,
        width: '20%',
    },
    {
        align: undefined,
        id: 'version',
        label: 'Version',
        sortable: false,
        truncate: false,
        width: '20%',
    },
    {
        align: undefined,
        id: 'status',
        label: 'Status',
        sortable: false,
        truncate: false,
        width: '16%',
    },
    {
        align: undefined,
        id: 'updated',
        label: 'Updated',
        sortable: false,
        truncate: false,
        width: '16%',
    },
    {
        align: 'right',
        id: 'deeplink',
        sortable: false,
        truncate: false,
        width: '5%',
    },
];

export const ExternalDeploymentsTable = memo(ExternalDeploymentsTableComponent);
