import React, { memo } from 'react';
import Moment from 'react-moment';
import upperFirst from 'lodash/upperFirst';
import camelCase from 'lodash/camelCase';
import deburr from 'lodash/deburr';
import { DotCard, DotCardContent, DotIconButton, DotPill } from '@digital-ai/dot-components';
import { ConnectionServer, ExternalDeployment as ExternalDeploymentType } from '../external-deployment.types';
import { IMG_SRC } from '../constants';
import { getDeepLink, getServerIcon, getVersionLabel } from '../helper/utils';
import './external-deployment.component.less';

export interface ExternalDeploymentProps {
    connectionServer: ConnectionServer;
    externalDeployment: ExternalDeploymentType;
}

const ExternalDeploymentComponent = ({
    connectionServer: { type, url },
    externalDeployment: { applicationUid, applicationPath, applicationName, state, statusGroup },
}: ExternalDeploymentProps) => {
    return (
        <DotCard className="external-deployment-card" key={applicationName}>
            <div data-testid={'app' + applicationName}>
                <DotCardContent>
                    <div className="icon-column">
                        <img alt={type} className="server-icon" src={getServerIcon(type)} />
                    </div>
                    <div className="application-name-column" title={applicationName}>
                        {applicationName}
                    </div>
                    <div className="environment-column">
                        <img alt="environment" className="server-icon" src={IMG_SRC.environment} />
                        <span title={state.destination}>{state.destination}</span>
                    </div>
                    <div className="version-column" title={state.versionTag.label}>
                        {getVersionLabel(type, state.versionTag.label)}
                    </div>
                    <div className="status-column">
                        <DotPill label={upperFirst(camelCase(deburr(state.deploymentStatus)))} size="small" status={statusGroup} />
                    </div>
                    <div className="last-updated-column">
                        <Moment fromNow interval={30000}>
                            {state.lastChangeTime}
                        </Moment>
                    </div>
                    <div className="link-column">
                        <DotIconButton
                            data-testid={`deeplink-${applicationUid}-${state.destination}`}
                            iconId="open-new-tab"
                            onClick={() => window.open(getDeepLink(url, applicationPath), '_blank')}
                            size="small"
                            tooltip="Open application"
                        />
                    </div>
                </DotCardContent>
            </div>
        </DotCard>
    );
};

export const ExternalDeployment = memo(ExternalDeploymentComponent);
