package com.xebialabs.xlrelease.scm.connector

import com.fasterxml.jackson.databind.exc.MismatchedInputException
import com.xebialabs.deployit.util.PasswordEncrypter
import com.xebialabs.xlrelease.domain.UserProfile
import com.xebialabs.xlrelease.domain.scm.connector.{GitHubOAuth2Token, GitHubScmConnectorConfig, GitHubUsernamePasswordCredential}
import com.xebialabs.xlrelease.scm.data.ValidatedCommitInfo
import org.kohsuke.github._

import java.net.HttpURLConnection.{HTTP_NOT_FOUND, HTTP_UNAUTHORIZED}
import scala.collection.mutable.ListBuffer
import scala.util.{Failure, Try}

class GitHubScmConnector(config: GitHubScmConnectorConfig) extends RestApiScmConnector(config) {
  private lazy val _repo: Try[GHRepository] = {
    val repo = for {
      gh <- tryConnect()
    } yield gh.getRepository(config.repository)
    repo.recoverWith {
      case e: GHFileNotFoundException => Failure(ScmException("Repository not found", e, HTTP_NOT_FOUND))
      case e: HttpException if e.getResponseCode == HTTP_UNAUTHORIZED => Failure(ScmException("Invalid credentials", e, HTTP_UNAUTHORIZED))
      case e: Exception => Failure(ScmException(e.getMessage))
    }
  }

  override protected def testRepository: Try[Unit] = _repo.map(_ => ())

  override protected def commitAndTag(blobs: ScmBlobs, commitInfo: ValidatedCommitInfo, user: UserProfile): Try[String] = {
    def createBranch(repo: GHRepository): Try[GHRef] = Try {
      val defaultBranch = repo.getRef(s"heads/${repo.getDefaultBranch}")
      repo.createRef(s"refs/heads/${config.branch}", defaultBranch.getObject.getSha)
    }

    for {
      repo <- _repo
      headRef <- Try(repo.getRef(s"heads/${config.branch}")).recoverWith {
        case ex: HttpException if ex.getResponseCode == 200 && ex.getCause.getCause.isInstanceOf[MismatchedInputException] =>
          createBranch(repo)
        case _: GHFileNotFoundException => createBranch(repo)
      }
      // 1. Get base tree
      baseTree <- Try(repo.getTree(config.branch))
      // 2. Upload blobs
      ghNamedBlobs <- uploadBlobs(blobs, repo)
      // 3. Create new tree with old as base
      newTree <- createTree(repo, baseTree.getSha, ghNamedBlobs)
      // 4. Commit changes
      commit <- Try(repo.createCommit().message(commitInfo.message).tree(newTree.getSha).parent(headRef.getObject.getSha).create())
      // 5. Create lightweight tag
      _ <- Try(repo.createRef(commitInfo.tag.toString, commit.getSHA1))
      // 6. Update HEAD
      _ <- Try(headRef.updateTo(commit.getSHA1))
    } yield commit.getSHA1
  }

  override protected def tagNotPresent(tag: String): Try[Boolean] = _repo.map(repo => Try(repo.getRef("tags/" + tag)).failed.isSuccess)

  private def tryConnect(): Try[GitHub] = Try {
    val builder = new GitHubBuilder()
      .withRateLimitHandler(GitHubRateLimitHandler.FAIL)
      .withAbuseLimitHandler(GitHubAbuseLimitHandler.FAIL)
      .withEndpoint(config.restApiUrl)

    config.credential match {
      case c: GitHubUsernamePasswordCredential => builder.withPassword(c.username, PasswordEncrypter.getInstance().ensureDecrypted(c.password))
      case c: GitHubOAuth2Token => builder.withOAuthToken(PasswordEncrypter.getInstance().ensureDecrypted(c.token))
      case null => throw new IllegalStateException(s"No credential provided for GitHub SCM config '${config.getTitle}'.")
    }

    builder.build()
  }

  private def uploadBlobs(blobs: ScmBlobs, repo: GHRepository): Try[List[(String, GHBlob)]] = Try {
    val ghBlobs = ListBuffer.empty[(String, GHBlob)]

    blobs.filesToAdd.foreach { file =>
      ghBlobs += file.absolutePath -> repo.createBlob().binaryContent(file.getContent()).create()
    }

    ghBlobs.toList
  }

  private def createTree(repo: GHRepository, baseTreeSha: String, blobs: Iterable[(String, GHBlob)]): Try[GHTree] = Try {
    val builder = repo.createTree().baseTree(baseTreeSha)
    blobs.foreach { case (filename, ghBlob) => builder.add(filename, ghBlob.getSha, false) }
    builder.create()
  }
}
