import React, { ChangeEvent } from 'react';
import { ReactWrapper } from 'enzyme';
import { Provider } from 'react-redux';
import moment from 'moment';
import { mockResizeObserver, mountWithStoreAndTheme } from '@xlr-ui/tests/unit/testing-utils';
import { headerDefinitions, RemoteRunnersTable, RemoteRunnersTableProps } from './remote-runners-table';
import { remoteRunners } from '../ducks/remote-runners.reducer';
import { remoteRunnerMock } from '../__mocks__/remote-runners.mocks';
import { DotChip, DotIcon, DotInputText, DotSwitch, DotTable, DotTableActions } from '@digital-ai/dot-components';
import { ChipGroup } from '@xlr-ui/app/react/components/chip-group/chip-group.component';

const { updateRemoteRunnerState } = remoteRunners.actions;

describe('RemoteRunnersTable component', () => {
    let wrapper: ReactWrapper;
    const dispatch = jest.fn();

    const defaultProps: RemoteRunnersTableProps = {
        viewPermission: true,
        editPermission: true,
        isLoading: false,
        onDeleteClick: jest.fn(),
        onEditClick: jest.fn(),
        onViewClick: jest.fn(),
        runners: [],
    };
    const mountComponent = (props: RemoteRunnersTableProps = defaultProps) => {
        wrapper = mountWithStoreAndTheme(<RemoteRunnersTable {...props} />, dispatch);
        expect(wrapper.find(Provider)).toExist();
    };

    const getStatusChip = (id: string) => wrapper.findWhere((node) => node.is(DotChip) && node.props()['data-testid'] === `status-chip-${id}`);
    const getTable = () => wrapper.find(DotTable);

    const getName = (index: number) => getTable().find('tbody tr').at(index).find('td').at(1);

    const getVersion = (index: number) => getTable().find('tbody tr').at(index).find('td').at(2);

    const getLastActivity = (index: number) => getTable().find('tbody tr').at(index).find('td').at(4);

    const getTokenExpiry = (index: number) => getTable().find('tbody tr').at(index).find('td').at(5);

    const getStateSwitch = (id: string) => wrapper.findWhere((node) => node.is(DotSwitch) && node.props()['data-testid'] === `state-switch-${id}`);

    const getTokenExpiryIcon = (id: string) => wrapper.findWhere((node) => node.is(DotIcon) && node.props()['data-testid'] === `expiry-icon-${id}`);

    const getTableActions = (id: string) => wrapper.findWhere((node) => node.is(DotTableActions) && node.props().id === id);

    const getSearch = () => wrapper.find('.remote-runners-table-search').findWhere((node) => node.is(DotInputText));

    beforeAll(() => {
        mockResizeObserver();
    });

    afterEach(() => {
        wrapper.unmount();
    });

    it('should render empty runners table', () => {
        mountComponent();

        const search = getSearch();
        expect(search).toExist();

        const searchProps = search.props();
        expect(searchProps.placeholder).toBe('Filter by name');

        const table = getTable();
        expect(table).toExist();

        const tableProps = table.props();
        expect(tableProps.columns).toStrictEqual(headerDefinitions);
        expect(tableProps.data.length).toBe(0);
        expect(tableProps.loading).toBe(false);
        expect(tableProps.rowsPerPage).toBe(10);
    });

    it('should render runners table with 1 runner', () => {
        mountComponent({ ...defaultProps, runners: [remoteRunnerMock] });

        const table = getTable();
        expect(table).toExist();

        const tableProps = table.props();
        expect(tableProps.columns).toStrictEqual(headerDefinitions);
        expect(tableProps.data.length).toBe(1);
        expect(tableProps.loading).toBe(false);
        expect(tableProps.rowsPerPage).toBe(10);

        const statusChip = getStatusChip(remoteRunnerMock.id);
        expect(statusChip).toExist();

        const statusChipProps = statusChip.props();
        expect(statusChipProps.className).toBe('status-success');
        expect(statusChipProps.error).toBe(false);
        expect(statusChipProps.isDeletable).toBe(false);
        expect(statusChipProps.children).toBe('Running');

        const nameLabel = getName(0);
        expect(nameLabel.text()).toBe('new-runner');

        const versionLabel = getVersion(0);
        expect(versionLabel.text()).toBe('0.2.0');

        const capabilitiesChips = table.find(ChipGroup);
        expect(capabilitiesChips).toExist();
        expect(capabilitiesChips.props().labels).toStrictEqual(['remote']);

        const lastActivityLabel = getLastActivity(0);
        expect(lastActivityLabel.text()).toBe('Never');

        const tokenExpiry = getTokenExpiry(0);
        expect(tokenExpiry.text()).toBe('Never');

        const stateSwitch = getStateSwitch(remoteRunnerMock.id);
        expect(stateSwitch).toExist();

        const stateSwitchProps = stateSwitch.props();
        expect(stateSwitchProps.checked).toBe(true);
        expect(stateSwitchProps.label).toBe('Enabled');

        const tableActions = getTableActions(`runner-actions-${remoteRunnerMock.id}`);
        expect(tableActions).toExist();
    });

    it('should render runner with disconnected status', () => {
        mountComponent({ ...defaultProps, runners: [{ ...remoteRunnerMock, available: false }] });

        const statusChip = getStatusChip(remoteRunnerMock.id);
        expect(statusChip).toExist();

        const statusChipProps = statusChip.props();
        expect(statusChipProps.className).toBe('status-error');
        expect(statusChipProps.error).toBe(true);
        expect(statusChipProps.isDeletable).toBe(false);
        expect(statusChipProps.children).toBe('Disconnected');
    });

    it('should render runner with no token expiry warning or error', () => {
        mountComponent({ ...defaultProps, runners: [{ ...remoteRunnerMock, tokenExpiryDate: moment().add(8, 'day').toDate() }] });

        const tokenExpiry = getTokenExpiry(0);
        expect(tokenExpiry).toExist();
        expect(tokenExpiry.text()).toBe('in 8 days');

        const tokenExpiryIcon = getTokenExpiryIcon(remoteRunnerMock.id);
        expect(tokenExpiryIcon).not.toExist();
    });

    it('should render runner with token expiry warning', () => {
        mountComponent({ ...defaultProps, runners: [{ ...remoteRunnerMock, tokenExpiryDate: moment().add(4, 'day').toDate() }] });

        const tokenExpiryIcon = getTokenExpiryIcon(remoteRunnerMock.id);
        expect(tokenExpiryIcon).toExist();
        expect(tokenExpiryIcon.getElement().props.iconId).toBe('warning-solid');
    });

    it('should render runner with token expiry error', () => {
        mountComponent({ ...defaultProps, runners: [{ ...remoteRunnerMock, tokenExpiryDate: moment().subtract(1, 'day').toDate() }] });

        const tokenExpiryIcon = getTokenExpiryIcon(remoteRunnerMock.id);
        expect(tokenExpiryIcon).toExist();
        expect(tokenExpiryIcon.getElement().props.iconId).toBe('error-solid');
    });

    it('should filter table data when search input is filled', () => {
        mountComponent({ ...defaultProps, runners: [remoteRunnerMock] });

        let table = getTable();
        expect(table).toExist();
        expect(table.props().data.length).toBe(1);

        const search = getSearch();
        expect(search).toExist();

        const e = { target: { value: 'non matching criteria' } } as unknown as ChangeEvent<HTMLInputElement>;
        search.invoke('onChange')?.(e);
        wrapper.update();

        table = getTable();
        expect(table).toExist();
        expect(table.props().data.length).toBe(0);
    });

    it('should run update runner saga when state switch is clicked', () => {
        mountComponent({ ...defaultProps, runners: [remoteRunnerMock] });

        const table = getTable();
        expect(table).toExist();
        expect(table.props().data.length).toBe(1);

        const stateSwitch = wrapper.find(DotSwitch);
        expect(stateSwitch).toExist();

        stateSwitch.invoke('onChange')?.({ target: { checked: false } } as never);
        expect(dispatch).toHaveBeenCalledWith(updateRemoteRunnerState({ id: remoteRunnerMock.id, enable: false }));
    });

    it('should render view action only when user has view permission but no edit permission', () => {
        mountComponent({ ...defaultProps, viewPermission: true, editPermission: false, runners: [remoteRunnerMock] });

        const tableActions = getTableActions(`runner-actions-${remoteRunnerMock.id}`);
        expect(tableActions).toExist();

        const actions = tableActions.props().actions;
        expect(actions.length).toBe(1);
        expect(actions[0].id).toBe('view');
        expect(actions[0].label).toBe('View runner');
    });

    it('should render edit and delete actions when user has edit permission', () => {
        mountComponent({ ...defaultProps, viewPermission: true, editPermission: true, runners: [remoteRunnerMock] });

        const tableActions = getTableActions(`runner-actions-${remoteRunnerMock.id}`);
        expect(tableActions).toExist();

        const actions = tableActions.props().actions;
        expect(actions.length).toBe(2);
        expect(actions[0].id).toBe('edit');
        expect(actions[0].label).toBe('Edit runner');
        expect(actions[1].id).toBe('delete');
        expect(actions[1].label).toBe('Delete runner');
    });

    it('should call onViewClick when view action is clicked', () => {
        const onViewClick = jest.fn();
        mountComponent({ ...defaultProps, viewPermission: true, editPermission: false, onViewClick, runners: [remoteRunnerMock] });

        const tableActions = getTableActions(`runner-actions-${remoteRunnerMock.id}`);
        const actions = tableActions.props().actions;

        actions[0].onClick();
        expect(onViewClick).toHaveBeenCalledWith(remoteRunnerMock);
    });
});
