import React, { ReactElement } from 'react';
import { ReactWrapper } from 'enzyme';
import { Provider } from 'react-redux';
import { act } from 'react-dom/test-utils';
import { mockResizeObserver, mountWithStoreAndTheme } from '@xlr-ui/tests/unit/testing-utils';
import { RemoteRunnersTableContainer, RemoteRunnersTableContainerProps } from './remote-runners-table-container';
import { RemoteRunnersTable } from './remote-runners-table';
import { initialState } from '../ducks/remote-runners.reducer';
import { remoteRunnerMock } from '../__mocks__/remote-runners.mocks';
import { ConnectionDialog } from '@xlr-ui/app/features/tasks/components/rails/common/connection-dialog/connection-dialog.component';
import { DotConfirmationDialog, DotTableActions, DotButton } from '@digital-ai/dot-components';
import { initialState as taskDrawerInitialState } from '@xlr-ui/app/features/tasks/ducks/task-drawer-connection.reducer';
import { initializedConfigurationStateMock } from '@xlr-ui/app/features/configuration/components/instance/__mocks__/configuration-instance-form.mock';
import { RootState } from '@xlr-ui/app/js/store.types';

describe('RemoteRunnersTableContainer component', () => {
    let wrapper: ReactWrapper;
    const dispatch = jest.fn();

    const defaultState: RootState = {
        remoteRunners: {
            ...initialState,
            remoteRunners: [remoteRunnerMock],
            userHasViewRunners: true,
            userHasEditRunners: true,
        },
        taskDrawerConnection: {
            ...taskDrawerInitialState,
        },
        configuration: {
            ...initializedConfigurationStateMock,
        },
    };

    const defaultProps: RemoteRunnersTableContainerProps = {
        isLoading: false,
        onViewChange: jest.fn(),
        runners: [remoteRunnerMock],
    };

    const mountComponent = (state: RootState = defaultState, props: RemoteRunnersTableContainerProps = defaultProps) => {
        wrapper = mountWithStoreAndTheme(<RemoteRunnersTableContainer {...props} />, dispatch, state);
        expect(wrapper.find(Provider)).toExist();
    };

    const getTableActions = (id: string) => wrapper.findWhere((node) => node.is(DotTableActions) && node.props().id === id);

    beforeAll(() => {
        mockResizeObserver();
    });

    afterEach(() => {
        wrapper.unmount();
    });

    it('should render remote runners table', () => {
        mountComponent();
        expect(wrapper.find(RemoteRunnersTable)).toExist();
    });

    it('should pass correct permissions to RemoteRunnersTable', () => {
        mountComponent();
        const table = wrapper.find(RemoteRunnersTable);
        expect(table.prop('viewPermission')).toBe(true);
        expect(table.prop('editPermission')).toBe(true);
    });

    it('should render remote runner connection dialog after click on edit button', () => {
        mountComponent();
        expect(wrapper.find(RemoteRunnersTable)).toExist();

        const tableActions = getTableActions(`runner-actions-${remoteRunnerMock.id}`);
        expect(tableActions).toExist();

        const actions = tableActions.props().actions as Array<{ iconId: string; id: string; label: string; onClick: () => void }>;
        const editAction = actions.find((action) => action.id === 'edit');
        expect(editAction).toBeDefined();

        act(() => {
            editAction!.onClick();
        });
        wrapper.update();

        expect(wrapper.find(ConnectionDialog)).toExist();
    });

    it('should render remote runner view dialog after click on view button', () => {
        const stateWithViewOnly: RootState = {
            ...defaultState,
            remoteRunners: {
                ...defaultState.remoteRunners,
                userHasEditRunners: false,
                userHasViewRunners: true,
            },
        };
        mountComponent(stateWithViewOnly);
        expect(wrapper.find(RemoteRunnersTable)).toExist();

        const tableActions = getTableActions(`runner-actions-${remoteRunnerMock.id}`);
        expect(tableActions).toExist();

        const actions = tableActions.props().actions as Array<{ iconId: string; id: string; label: string; onClick: () => void }>;
        const viewAction = actions.find((action) => action.id === 'view');
        expect(viewAction).toBeDefined();

        act(() => {
            viewAction!.onClick();
        });
        wrapper.update();

        expect(wrapper.find(ConnectionDialog)).toExist();
        expect(wrapper.find(ConnectionDialog).prop('readOnly')).toBe(true);
    });

    it('should render remote runner delete dialog after click on delete button', () => {
        mountComponent();
        expect(wrapper.find(RemoteRunnersTable)).toExist();

        const tableActions = getTableActions(`runner-actions-${remoteRunnerMock.id}`);
        expect(tableActions).toExist();

        const actions = tableActions.props().actions as Array<{ iconId: string; id: string; label: string; onClick: () => void }>;
        const deleteAction = actions.find((action) => action.id === 'delete');
        expect(deleteAction).toBeDefined();

        act(() => {
            deleteAction!.onClick();
        });
        wrapper.update();

        expect(wrapper.find(DotConfirmationDialog)).toExist();
    });

    it('should not render Add Runner button when user has no edit permission', () => {
        const stateWithViewOnly: RootState = {
            ...defaultState,
            remoteRunners: {
                ...defaultState.remoteRunners,
                userHasEditRunners: false,
                userHasViewRunners: true,
            },
        };
        mountComponent(stateWithViewOnly);

        const actionToolbarPortal = wrapper.find('ActionToolbarPortalContainer');
        expect(actionToolbarPortal.props().children).toBeFalsy();
    });

    it('should render Add Runner button when user has edit permission', () => {
        mountComponent();

        const actionToolbarPortal = wrapper.find('ActionToolbarPortalContainer');
        const buttonElement = actionToolbarPortal.props().children as ReactElement;

        expect(buttonElement).toBeTruthy();
        expect(buttonElement.type).toBe(DotButton);
        expect(buttonElement.props.children).toBe('Add Runner');
    });
});
