import React, { useState } from 'react';
import { DotButton, DotConfirmationDialog, DotIcon } from '@digital-ai/dot-components';
import { remoteRunners } from '../ducks/remote-runners.reducer';
import { RemoteRunner } from '../remote-runners.types';
import { useAppDispatch } from '@xlr-ui/app/js/hooks';
import { ConnectionDialog } from '@xlr-ui/app/features/tasks/components/rails/common/connection-dialog/connection-dialog.component';
import { RemoteRunnersTable } from './remote-runners-table';
import { ActionToolbarPortalContainer } from '@xlr-ui/app/features/main-navigation/action-toolbar/components/action-toolbar.portal';
import './remote-runners-table-container.less';

export interface RemoteRunnersTableContainerProps {
    isLoading: boolean;
    onViewChange: () => void;
    runners: RemoteRunner[];
}

const { reloadRemoteRunners, deleteRemoteRunner } = remoteRunners.actions;

export const RemoteRunnersTableContainer = ({ isLoading, onViewChange, runners }: RemoteRunnersTableContainerProps) => {
    const dispatch = useAppDispatch();

    const [selectedRunner, setSelectedRunner] = useState<RemoteRunner>();
    const [isEditMode, setEditMode] = useState<boolean>(false);
    const [isDeleteMode, setDeleteMode] = useState<boolean>(false);

    const handleEditClick = (remoteRunner: RemoteRunner) => {
        setSelectedRunner(remoteRunner);
        setEditMode(true);
    };

    const handleCloseEditModal = () => {
        setEditMode(false);
    };

    const handleDeleteClick = (remoteRunner: RemoteRunner) => {
        setSelectedRunner(remoteRunner);
        setDeleteMode(true);
    };

    const handleCloseDeleteModal = () => {
        setDeleteMode(false);
    };

    const handleDeleteSubmit = () => {
        if (selectedRunner) {
            dispatch(deleteRemoteRunner(selectedRunner));
            setDeleteMode(false);
        }
    };

    return (
        <>
            <ActionToolbarPortalContainer>
                <DotButton onClick={onViewChange}>Add Runner</DotButton>
            </ActionToolbarPortalContainer>
            {selectedRunner && isEditMode && (
                <ConnectionDialog
                    configurationSaveAction={reloadRemoteRunners}
                    connectionId={selectedRunner.id}
                    hiddenProperties={['available', 'publicKey', 'encryptJobData']}
                    isNewInstance={false}
                    keyOrName={selectedRunner.id}
                    onCancel={handleCloseEditModal}
                    onSave={handleCloseEditModal}
                    referencedType="xlRelease.RemoteJobRunner"
                    showTestButton={false}
                />
            )}
            {selectedRunner && isDeleteMode && (
                <DotConfirmationDialog
                    message={
                        <>
                            <DotIcon className="warning-delete-runner-icon" iconId="warning-solid" />
                            This will remove the Runner from the Release overview, but not uninstall it.
                            <br />
                            To uninstall the Runner, follow the instructions on the documentation.
                            <br />
                            If not uninstalled, the Runner will return to this screen when it becomes active.
                            <br /> <br />
                            Are you sure you want to delete <strong>{selectedRunner.title}</strong> Runner?
                        </>
                    }
                    onCancel={handleCloseDeleteModal}
                    onSubmit={handleDeleteSubmit}
                    open={isDeleteMode}
                    submitButtonProps={{
                        label: 'Delete',
                        type: 'destructive',
                        startIcon: <DotIcon iconId="delete" />,
                    }}
                    title="Delete Runner"
                />
            )}
            <RemoteRunnersTable isLoading={isLoading} onDeleteClick={handleDeleteClick} onEditClick={handleEditClick} runners={runners} />
        </>
    );
};
