package com.xebialabs.xlrelease.runner.impl

import com.fasterxml.jackson.annotation.JsonIgnoreProperties
import com.fasterxml.jackson.databind.ObjectMapper
import com.fasterxml.jackson.databind.annotation.JsonDeserialize
import com.fasterxml.jackson.databind.deser.std.UntypedObjectDeserializer
import com.fasterxml.jackson.module.scala.DefaultScalaModule
import com.xebialabs.deployit.plugin.api.reflect.{PropertyDescriptor, PropertyKind, Type}
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.xlrelease.actors.ReleaseActorService
import com.xebialabs.xlrelease.domain.facet.TaskReportingRecord
import com.xebialabs.xlrelease.domain.runner.RemoteJobRunner
import com.xebialabs.xlrelease.domain.utils.ReleaseCloneHelper
import com.xebialabs.xlrelease.domain.{Configuration, ContainerTask, RemoteExecution, Task}
import com.xebialabs.xlrelease.repository.TaskRepository
import com.xebialabs.xlrelease.runner.domain._
import com.xebialabs.xlrelease.runner.impl.RunnerJobServiceImpl._
import com.xebialabs.xlrelease.runner.impl.RunnerScriptService.ErrorMessageKey
import com.xebialabs.xlrelease.runner.service.RunnerJobService
import com.xebialabs.xlrelease.scheduler._
import com.xebialabs.xlrelease.scheduler.logs.TaskExecutionLogService
import com.xebialabs.xlrelease.script.{EncryptionHelper, FailureContainerTaskResult, SuccessContainerTaskResult}
import com.xebialabs.xlrelease.security.UsernamePassword
import com.xebialabs.xlrelease.security.authentication.AuthenticationService
import com.xebialabs.xlrelease.serialization.json.jackson.CiDeserializer
import com.xebialabs.xlrelease.serialization.json.repository.ResolveOptions
import com.xebialabs.xlrelease.service._
import com.xebialabs.xlrelease.storage.domain.LogEntry
import com.xebialabs.xlrelease.support.serialization.SerializableMsg
import com.xebialabs.xlrelease.utils.DateVariableUtils
import grizzled.slf4j.Logging
import org.apache.commons.lang3.exception.ExceptionUtils
import org.springframework.stereotype.Component
import org.springframework.util.StringUtils.hasText

import java.util
import java.util.{Collections, Date, UUID}
import scala.annotation.meta.field
import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

@Component
class RunnerJobServiceImpl(jobQueue: JobQueue,
                           releaseActorService: ReleaseActorService,
                           val configurationVariableService: ConfigurationVariableService,
                           facetService: FacetService,
                           commentService: CommentService,
                           taskRepository: TaskRepository,
                           val configurationService: ConfigurationService,
                           taskExecutionLogService: TaskExecutionLogService,
                           runnerScriptService: RunnerScriptService,
                           val authenticationService: AuthenticationService
                          )
  extends RunnerJobService
    with JobToJobDataConversion
    with OutputContextDeserialization
    with Logging {

  private lazy val _objectMapper: ObjectMapper = {
    val mapper = new ObjectMapper()
    mapper.registerModule(DefaultScalaModule)
    mapper
  }

  override def objectMapper: ObjectMapper = _objectMapper

  override def reserveJob(runnerId: RunnerId): Option[JobData] = {
    val taskJob = jobQueue.get(runnerId)
    taskJob.flatMap(toJobData(runnerId))
  }

  override def confirmJob(runnerId: RunnerId, jobId: JobId): Boolean = {
    jobQueue.confirm(runnerId, jobId)
  }

  //noinspection ScalaStyle
  override def finishJob(jobResult: JobResult): Unit = {
    jobResult match {
      case result: ScriptJobResult =>
        result match {
          case _: ScriptJobSuccess =>
            val outputContext = deserialize(result.runnerId, result.output)
            runnerScriptService.scriptExecuted(result.executionId,  outputContext.outputProperties)
          case scriptJobFailure: ScriptJobFailure =>
            val payload: util.Map[String, AnyRef] = new util.HashMap[String, AnyRef]()
            payload.put(ErrorMessageKey, scriptJobFailure.errorMessage)
            runnerScriptService.scriptExecuted(result.executionId, payload)
        }
      case result: TaskJobResult =>
        jobQueue.finish(result.jobId)
        val taskResult = Try {
          val outputContext = deserialize(result.runnerId, result.output)
          result match {
            case TaskJobSuccess(taskId, jobId, _, containerStatusCode, _) =>
              if (containerStatusCode == 0 && outputContext.exitCode == 0) {
                createTaskReportingRecords(taskId, outputContext.reportingRecords)
                SuccessContainerTaskResult(taskId, outputContext.outputProperties)
              } else {
                logger.trace(s"Failing the container task [$taskId]. " +
                  s"Job [$jobId] exited with container status code [$containerStatusCode] and exit code [${outputContext.exitCode}].")
                val reason = extractErrorMessage(outputContext).getOrElse("Job or container did not terminate with exit code 0.")
                FailureContainerTaskResult(taskId, outputContext.outputProperties, reason)
              }
            case TaskJobFailure(taskId, _, _, _, errorMessage) =>
              FailureContainerTaskResult(taskId, outputContext.outputProperties, errorMessage)
          }
        }.recover {
          case ex: Exception =>
            logger.error("Failed to process supplied job result", ex)
            FailureContainerTaskResult(result.taskId, Collections.emptyMap(), ExceptionUtils.getRootCauseMessage(ex))
        }.get
        // TODO make this finish method reliable
        releaseActorService.finishContainerTask(taskResult)
      case jobResult =>
        logger.warn(s"Can't handle supplied job result of type [${jobResult.getClass.getSimpleName}]")
    }
    updateRunnerLastActivity(jobResult.runnerId)
  }

  private def extractErrorMessage(outputContext: JobOutputContext): Option[String] = {
    Option(outputContext.jobErrorMessage).filter(hasText)
  }

  override def log(logEntry: LogEntry): Unit = {
    taskExecutionLogService.log(logEntry)
  }

  override def executeDirectives(directives: Seq[JobDirective]): Unit = {
    // TODO maybe extract directive parser/processors?
    val directivesOrderedByTaskId = directives.groupBy(_.taskId).values.flatten
    val groupedDirectives = directivesOrderedByTaskId.groupBy(_.directiveName)
    groupedDirectives.collect {
      case ("status", directives) =>
        for {
          d <- directives.lastOption
        } releaseActorService.updateTaskStatusLine(d.taskId, d.payload)
      case ("comment", directives) =>
        for {
          d <- directives
        } createComment(d.taskId, d.payload)
      case (directiveName, _) => logger.warn(s"Received unknown directive [$directiveName]")
    }
  }


  private def createComment(taskId: String, comment: String): Unit = {
    val task: Task = taskRepository.findById(taskId, ResolveOptions.WITHOUT_DECORATORS)
    commentService.appendComment(task, null, comment)
  }

  private def createTaskReportingRecords(taskId: String, reportingRecords: util.List[_ <: TaskReportingRecord]): Unit = {
    if (null != reportingRecords) {
      Try(reportingRecords.asScala.foreach { record =>
        val task: Task = taskRepository.findById(record.getTargetId, ResolveOptions.WITHOUT_DECORATORS)
        record.setCreatedViaApi(true)
        record.setRetryAttemptNumber(task.getFailuresCount)

        facetService.addTaskReportingRecord(record, applyTaskAttributes = true)
      }).recover {
        case ex: Exception => logger.error(s"Unable to add task reporting record for task [$taskId]", ex)
      }
    }
  }

  private def updateRunnerLastActivity(runnerId: RunnerId, date: Date = new Date()): Unit = {
    val remoteRunner = configurationService.read(runnerId).asInstanceOf[RemoteJobRunner]
    remoteRunner.setLastActivity(date)
    configurationService.createOrUpdate(remoteRunner)
  }
}


private[impl] trait InputContextFactory {
  def create(task: ContainerTask, usernamePassword: UsernamePassword): JobInputContext = {
    val release = task.getRelease
    val releaseContext = ReleaseContext(
      release.getId,
      AutomatedTaskAsUserContext(
        usernamePassword.username,
        usernamePassword.password
      )
    )
    val taskContext: TaskContext = createTaskInputContext(Some(task.getTaskType.toString), task)
    JobInputContext(releaseContext, taskContext)
  }

  private def createTaskInputContext(taskTypeName: Option[String], remoteExecution: ConfigurationItem) = {
    import scala.jdk.CollectionConverters._
    val taskDescriptor = remoteExecution.getType.getDescriptor
    val pds = taskDescriptor.getPropertyDescriptors.asScala.filter(pd => Set(Task.CATEGORY_INPUT, Task.CATEGORY_OUTPUT, "script").contains(pd.getCategory))
    val taskProperties = pds.map(pd => PropertyDefinition(pd.getName, propertyValue(remoteExecution, pd), pd.getKind.toString, pd.getCategory, pd.isPassword))
    val taskContext = TaskContext(
      remoteExecution.getId,
      taskTypeName.orNull,
      taskProperties.toList
    )
    taskContext
  }

  def createForScript(remoteExecution: RemoteExecution): JobInputContext = {
    JobInputContext(null, createTaskInputContext(Option(remoteExecution.getType.toString), remoteExecution))
  }

  //noinspection ScalaStyle
  private def propertyValue(ci: ConfigurationItem, pd: PropertyDescriptor): AnyRef = {
    import java.util.{List => JList, Map => JMap, Set => JSet}
    import scala.jdk.CollectionConverters._
    val rawValue = pd.get(ci)
    if (rawValue != null) {
      pd.getKind match {
        case PropertyKind.BOOLEAN => rawValue
        case PropertyKind.INTEGER => rawValue
        case PropertyKind.STRING => rawValue
        case PropertyKind.ENUM => rawValue.toString
        case PropertyKind.DATE => DateVariableUtils.printDate(rawValue.asInstanceOf[Date])
        case PropertyKind.CI => ciValue(ci, pd)
        case PropertyKind.SET_OF_CI =>
          val ciCollection = rawValue.asInstanceOf[JSet[ConfigurationItem]]
          ciCollection.asScala.map(c => CiReference(c.getId, c.getType.toString))
        case PropertyKind.LIST_OF_CI =>
          val ciCollection = rawValue.asInstanceOf[JList[ConfigurationItem]]
          ciCollection.asScala.map(c => CiReference(c.getId, c.getType.toString))
        case PropertyKind.SET_OF_STRING => rawValue.asInstanceOf[JSet[String]].asScala.toSet
        case PropertyKind.LIST_OF_STRING => rawValue.asInstanceOf[JList[String]].asScala.toList
        case PropertyKind.MAP_STRING_STRING => rawValue.asInstanceOf[JMap[String, String]].asScala.toMap
      }
    } else {
      null
    }
  }

  private def ciValue(ci: ConfigurationItem, pd: PropertyDescriptor): AnyRef = {
    import scala.jdk.CollectionConverters._
    val rawValue = pd.get(ci).asInstanceOf[ConfigurationItem]
    val isConfiguration = pd.getReferencedType.isSubTypeOf(Type.valueOf(classOf[Configuration]))
    val isContainerTask = pd.getReferencedType.isSubTypeOf(Type.valueOf(classOf[ContainerTask]))
    if (isConfiguration || isContainerTask) {
      EncryptionHelper.decrypt(rawValue)
      val ciDescriptor = rawValue.getType.getDescriptor
      val pds = ciDescriptor.getPropertyDescriptors.asScala
      val ciProperties = pds.map(pd => PropertyDefinition(pd.getName, propertyValue(rawValue, pd), pd.getKind.toString, pd.getCategory, pd.isPassword))
      ConfigurationContext(
        rawValue.getId,
        rawValue.getType.toString,
        ciProperties.toList
      )
    } else {
      CiReference(rawValue.getId, rawValue.getType.toString) // is it embedded, nested or what, is it a CI reference, like a container CI?
    }

  }
}

private[impl] trait InputContextSerialization extends Logging {

  def objectMapper: ObjectMapper

  def configurationService: ConfigurationService

  def serialize(inputContext: JobInputContext, runnerId: RunnerId): ContextData = {
    val inputContextJson = objectMapper.writeValueAsString(inputContext)
    encryptContextData(runnerId, inputContextJson)
  }

  private def encryptContextData(runnerId: RunnerId, inputContextJson: String): ContextData = {
    val config = configurationService.read(runnerId)
    config match {
      case runner: RemoteJobRunner =>
        ContextDataHelper.encryptData(runner, inputContextJson)
      case _ => PlainContextData(inputContextJson)
    }
  }
}

private[impl] trait OutputContextDeserialization extends Logging {

  def objectMapper: ObjectMapper

  def configurationService: ConfigurationService

  def deserialize(runnerId: RunnerId, output: ContextData): JobOutputContext = {
    def readValue(plainData: PlainContextData): JobOutputContext = {
      val data = plainData.data
      if (hasText(data)) {
        objectMapper.readValue(plainData.data, classOf[JobOutputContext])
      } else {
        JobOutputContext(-1, "", Collections.emptyMap(), Collections.emptyList())
      }
    }

    output match {
      case plainData: PlainContextData =>
        readValue(plainData)
      case encryptedData: EncryptedContextData =>
        val config = configurationService.read(runnerId)
        config match {
          case runner: RemoteJobRunner =>
            val plainContextData = ContextDataHelper.decryptData(runner, encryptedData)
            readValue(plainContextData)
          case _ => throw new RuntimeException(s"Encrypted job data is only supported for remote runners")
        }
    }
  }
}

trait JobToJobDataConversion extends InputContextFactory with InputContextSerialization {

  def configurationVariableService: ConfigurationVariableService

  def authenticationService: AuthenticationService

  def toJobData(runnerId: RunnerId)(job: Job): Option[JobData] = {
    job match {
      case job: TaskJob[_] => taskJobData(runnerId)(job)
      case StopWorkerThread() => None
      case FailJob(_, _) => None
    }
  }

  private def taskJobData(runnerId: RunnerId)(taskJob: TaskJob[_]): Option[JobData] = {
    taskJob match {
      case job: ContainerTaskJob => toJobData(runnerId, job)
      case job => Some(SomeJobData(job.id))
    }
  }

  private def toJobData(runnerId: RunnerId, job: ContainerTaskJob): Option[ContainerJobData] = {
    Try {
      val task: ContainerTask = job.taskRef.get()
      // resolve variables
      val globalVariables = task.getRelease.getGlobalVariables
      configurationVariableService.resolveFromCi[ContainerTask](task, globalVariables)(ci => ci.getInputProperties.asScala)
      val usernamePassword = authenticationService.loginScriptUser(task)
      // decrypt task
      val containerTask = ReleaseCloneHelper.clone(task)
      EncryptionHelper.decrypt(containerTask.getRelease)
      EncryptionHelper.decrypt(containerTask)
      val inputContext = create(containerTask, usernamePassword)
      val inputContextData: ContextData = serialize(inputContext, runnerId)
      val abortTimeout = containerTask.getAbortTimeout
      val maxRetryAttempts = containerTask.getMaxRetryAttempts
      val retryDelay = containerTask.getRetryDelay
      val taskImg = containerTask.getImage
      ContainerJobData(job.id, task.getExecutionId, job.taskId, taskImg, abortTimeout, maxRetryAttempts, retryDelay, inputContextData)
    } match {
      case Failure(ex) =>
        logger.error("Unable to create job data", ex)
        None
      case Success(value) =>
        Some(value)
    }
  }
}

trait ContainerTaskToScriptJobDataConversion extends InputContextFactory with InputContextSerialization {
  def toScriptJobData(runnerId: RunnerId, remoteExecution: RemoteExecution): Option[ContainerScriptJobData] = {
    Try {
      EncryptionHelper.decrypt(remoteExecution)
      val inputContext = createForScript(remoteExecution)
      val inputContextData: ContextData = serialize(inputContext, runnerId)
      val abortTimeout = remoteExecution.getAbortTimeout
      val maxRetryAttempts = remoteExecution.getMaxRetryAttempts
      val retryDelay = remoteExecution.getRetryDelay
      val taskImg = remoteExecution.getImage
      ContainerScriptJobData(UUID.randomUUID().toString, taskImg, abortTimeout, maxRetryAttempts, retryDelay, inputContextData)
    } match {
      case Failure(ex) =>
        logger.error("Unable to create script job data", ex)
        None
      case Success(value) =>
        Some(value)
    }
  }
}

object RunnerJobServiceImpl {
  case class CiReference(id: String, `type`: String) extends SerializableMsg

  case class PropertyDefinition(name: String, value: AnyRef, kind: String, category: String, password: Boolean) extends SerializableMsg {
    override def toString: String = {
      val valueAsString = if (password) "****" else value
      s"PropertyDefinition(name = $name, value = $valueAsString, kind = $kind, category = $category, password = $password)"
    }
  }

  case class ConfigurationContext(id: String, `type`: String, properties: List[PropertyDefinition]) extends SerializableMsg

  case class TaskContext(id: String, `type`: String, properties: List[PropertyDefinition]) extends SerializableMsg

  case class AutomatedTaskAsUserContext(username: String, password: String) extends SerializableMsg {
    override def toString: String = {
      s"AutomatedTaskAsUserContext(username = $username)"
    }
  }

  case class ReleaseContext(id: String, automatedTaskAsUser: AutomatedTaskAsUserContext) extends SerializableMsg

  case class JobInputContext(release: ReleaseContext, task: TaskContext) extends SerializableMsg

  @JsonIgnoreProperties(ignoreUnknown = true)
  case class JobOutputContext(exitCode: Int,
                              jobErrorMessage: String = "",
                              @(JsonDeserialize @field)(contentUsing = classOf[UntypedObjectDeserializer])
                              outputProperties: util.Map[String, AnyRef] = new util.HashMap[String, AnyRef](),
                              @(JsonDeserialize @field)(contentUsing = classOf[CiDeserializer])
                              reportingRecords: util.List[_ <: TaskReportingRecord] = new util.ArrayList[TaskReportingRecord]())
}
