package com.xebialabs.xlrelease.runner.impl

import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.domain.runner.JobRunner
import com.xebialabs.xlrelease.repository.JobRunnerRepository
import com.xebialabs.xlrelease.runner.impl.JobRunnerService.{AliveThreshold, RemoteScriptRunnerCapability}
import com.xebialabs.xlrelease.runner.impl.RunnerProxyActor.{LastRunnerTime, LastRunnerTimeResponse}
import com.xebialabs.xlrelease.runner.impl.spring.RemoteJobRunnerConfiguration.RunnerProxyActorHolder
import org.apache.pekko.util.Timeout
import org.springframework.stereotype.Service

import scala.concurrent.Await
import scala.jdk.CollectionConverters._

object JobRunnerService {
  val RemoteScriptRunnerCapability = "remote-script"
  val AliveThreshold = 60000
}

@Service
class JobRunnerService(xlrConfig: XlrConfig,
                        jobRunnerRepository: JobRunnerRepository,
                        runnerProxyActorHolder: RunnerProxyActorHolder) {
  def findScriptExecutor(): Option[JobRunner] = {
    implicit val askTimeout: Timeout = xlrConfig.timeouts.releaseActorReceive

    val currentTime = System.currentTimeMillis()
    val jobRunners = jobRunnerRepository.findAll().asScala
      .filter(r => {
        r.getCapabilities.contains(RemoteScriptRunnerCapability) && r.isEnabled && r.isAvailable
      })

    jobRunners
      .map(r =>
        (r, try {
          Some(Await.result(
            runnerProxyActorHolder ? LastRunnerTime(r.getId),
            askTimeout.duration
          ).asInstanceOf[LastRunnerTimeResponse])
        } catch {
          case _: Exception => None
        })
      ).collect {
        case (runner, Some(last)) if last.timestamp.isDefined && currentTime - last.timestamp.get < AliveThreshold =>
          (runner, last)
      }.maxByOption(_._2.timestamp).map(_._1)
  }
}
