package com.xebialabs.xlrelease.runner.impl

import com.fasterxml.jackson.annotation.{JsonCreator, JsonProperty}
import com.fasterxml.jackson.core.`type`.TypeReference
import com.xebialabs.xlrelease.domain.Configuration
import com.xebialabs.xlrelease.domain.configuration.ContainerHttpConnection
import com.xebialabs.xlrelease.service.ConfigurationStatusService
import com.xebialabs.xlrelease.views.SharedConfigurationStatusResponse
import grizzled.slf4j.Logging
import org.springframework.stereotype.Component

import java.util
import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

@Component
class RemoteConfigurationStatusService(runnerScriptService: RunnerScriptService) extends ConfigurationStatusService with Logging {

  private val testConnectionScriptProperty = "testConnectionScript"

  override def supports(configuration: Configuration): Boolean = {
    configuration.hasProperty(testConnectionScriptProperty) && Option(configuration.getProperty[String](testConnectionScriptProperty)).exists(_.nonEmpty)
  }

  def checkStatus(configuration: Configuration): SharedConfigurationStatusResponse = {
    logger.debug(s"Executing remote test connection script for ${configuration.getTitle}")
    Try {
      val inputParameters: util.Map[String, AnyRef] = new util.HashMap[String, AnyRef]()
      inputParameters.put("server", configuration)
      val capabilities = configuration match {
        case containerHttpConnection: ContainerHttpConnection =>
          containerHttpConnection.getCapabilities.asScala.toSet
        case _ =>
          Set.empty[String]
      }
      val response = runnerScriptService.executeScript[RemoteConfigurationStatusResponse](
        configuration.getProperty[String](testConnectionScriptProperty),
        capabilities,
        inputParameters,
        new TypeReference[RemoteConfigurationStatusResponse] {}
      )

      if (response.success) {
        new SharedConfigurationStatusResponse(true, true)
      } else {
        new SharedConfigurationStatusResponse(false, true, response.output)
      }
    }
  } match {
    case Success(response) => response
    case Failure(e: Throwable) =>
      logger.error(s"Error while executing remote test connection script for ${configuration.getTitle} : ${e.getMessage}")
      new SharedConfigurationStatusResponse(false, false, e.getMessage)
  }

}

case class RemoteConfigurationStatusResponse @JsonCreator()(@JsonProperty("success") success: Boolean, @JsonProperty("output") output: String)
