package com.xebialabs.xlrelease.runner.impl

import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.domain.runner.JobRunner
import com.xebialabs.xlrelease.repository.JobRunnerRepository
import com.xebialabs.xlrelease.runner.domain.RunnerId
import com.xebialabs.xlrelease.runner.impl.RunnerProxyActor.{LastRunnerTime, LastRunnerTimeResponse}
import com.xebialabs.xlrelease.runner.impl.spring.RemoteJobRunnerConfiguration.RunnerProxyActorHolder
import com.xebialabs.xlrelease.service.ConfigurationService
import grizzled.slf4j.Logging
import org.apache.pekko.util.Timeout
import org.springframework.scheduling.annotation.Scheduled
import org.springframework.stereotype.Service

import scala.concurrent.Await
import scala.jdk.CollectionConverters._
import scala.util.Try

@Service
class RunnerAvailabilityService(xlrConfig: XlrConfig,
                                jobRunnerRepository: JobRunnerRepository,
                                runnerProxyActorHolder: RunnerProxyActorHolder,
                                configurationService: ConfigurationService) extends Logging {
  private val availableProperty = "available"

  def alive(runnerId: RunnerId): Unit = {
    jobRunnerRepository.findRunner(runnerId).foreach(runner => {
      if (!runner.isAvailable) {
        logger.info(s"Unavailable runner '${runner.getId}' confirmed it's alive again and will be marked as available")
        updateRunnerAvailability(runner, true)
      }
    })
  }

  @Scheduled(initialDelay = 60000, fixedRate = 60000)
  private def checkRunnersAvailability(): Unit = {
    val currentTime = System.currentTimeMillis()
    jobRunnerRepository.findAll().asScala
      .map(runner => mapRunnerToLastTimeAlive(runner))
      .collect { case (runner, Some(last)) if isRunnerExpired(runner, last, currentTime) => runner }
      .foreach(runner => {
        logger.warn(s"Runner '${runner.getId}' failed to confirm it's alive for more than ${runner.idleTimeToLive} seconds and will be marked as unavailable")
        updateRunnerAvailability(runner, false)
      })
  }

  private def isRunnerExpired(runner: JobRunner, last: LastRunnerTimeResponse, currentTime: Long) = {
    runner.isAvailable && last.timestamp.isDefined && runner.idleTimeToLive > 0 && currentTime - last.timestamp.get > runner.idleTimeToLive * 1000
  }

  private def mapRunnerToLastTimeAlive(runner: JobRunner) = {
    implicit val askTimeout: Timeout = xlrConfig.timeouts.releaseActorReceive
    val lastTimeAlive = Try(Await.result(
      runnerProxyActorHolder ? LastRunnerTime(runner.getId),
      askTimeout.duration
    ).asInstanceOf[LastRunnerTimeResponse]).toOption
    (runner, lastTimeAlive)
  }

  private def updateRunnerAvailability(runner: JobRunner, available: Boolean): Unit = {
    runner.setProperty(availableProperty, available)
    configurationService.createOrUpdate(runner)
  }
}

