package com.xebialabs.xlrelease.runner.impl

import com.xebialabs.xlrelease.config.CacheManagementConstants.RUNNER_CACHE_MANAGER_NAME
import com.xebialabs.xlrelease.domain.events.ConfigurationDeletedEvent
import com.xebialabs.xlrelease.domain.runner.RemoteJobRunner
import com.xebialabs.xlrelease.events.{EventListener, Subscribe}
import com.xebialabs.xlrelease.runner.domain.{RunnerId, RunnerToken}
import com.xebialabs.xlrelease.runner.impl.spring.RunnerCacheConfiguration.RUNNER_TOKEN_CACHE_NAME
import com.xebialabs.xlrelease.runner.repository.RunnerTokenRepository
import com.xebialabs.xlrelease.runner.repository.data.RunnerTokenRow
import com.xebialabs.xlrelease.runner.service.RunnerTokenService
import com.xebialabs.xlrelease.service.BroadcastService
import com.xebialabs.xlrelease.utils.{TokenGenerator, TokenTypes}
import grizzled.slf4j.Logging
import io.micrometer.core.annotation.Timed
import org.springframework.cache.annotation.{CacheConfig, CacheEvict, Cacheable}
import org.springframework.stereotype.Service

import java.util.Date

@Service
@EventListener
@CacheConfig(cacheManager = RUNNER_CACHE_MANAGER_NAME)
class RunnerTokenServiceImpl(broadcastService: BroadcastService,
                             runnerTokenRepository: RunnerTokenRepository)
  extends RunnerTokenService with Logging {

  @Timed
  override def createOrUpdateRunnerToken(runnerId: RunnerId, expiryDate: Date): RunnerToken = {
    val token: String = TokenGenerator.generate(TokenTypes.RRA)
    val tokenHash: String = TokenGenerator.hash(token)

    if (runnerTokenRepository.exists(runnerId)) {
      runnerTokenRepository.update(runnerId, tokenHash, expiryDate)
      broadcastService.broadcast(RunnerTokenUpdatedEvent(runnerId), publishEventOnSelf = true)
    } else {
      runnerTokenRepository.create(runnerId, tokenHash, expiryDate)
      broadcastService.broadcast(RunnerTokenCreatedEvent(runnerId), publishEventOnSelf = true)
    }
    RunnerToken(runnerId, token)
  }

  @Timed
  override def deleteRunnerToken(runnerId: RunnerId): Unit = {
    runnerTokenRepository.delete(runnerId)
    broadcastService.broadcast(RunnerTokenDeletedEvent(runnerId), publishEventOnSelf = true)
  }

  @Timed
  @Cacheable(cacheNames = Array(RUNNER_TOKEN_CACHE_NAME))
  override def findByToken(tokenHash: String): Option[RunnerTokenRow] = {
    runnerTokenRepository.findByToken(tokenHash)
  }

  @Subscribe
  @CacheEvict(cacheNames = Array(RUNNER_TOKEN_CACHE_NAME), allEntries = true)
  def onRunnerTokenEvent(runnerTokenEvent: RunnerTokenEvent): Unit = {
    logger.debug("Invalidating runner token cache")
  }

  @Subscribe
  def onJobRunnerDeleted(event: ConfigurationDeletedEvent): Unit = {
    val config = event.conf
    config match {
      case runner: RemoteJobRunner =>
        logger.trace(s"Deleting all tokens associated with runner [${runner.getId}]")
        deleteRunnerToken(runner.getId)
      case _ => () //Nothing
    }
  }

}
