package com.xebialabs.xlrelease.runner.impl

import com.xebialabs.xlplatform.cluster.ClusterMode.{FULL, HOT_STANDBY, STANDALONE}
import com.xebialabs.xlrelease.actors.ActorSystemHolder
import com.xebialabs.xlrelease.actors.initializer.ActorInitializer
import com.xebialabs.xlrelease.config.XlrConfig
import grizzled.slf4j.Logging
import org.apache.pekko.actor.{ActorRef, PoisonPill, Props}
import org.apache.pekko.cluster.singleton.{ClusterSingletonManager, ClusterSingletonManagerSettings, ClusterSingletonProxy, ClusterSingletonProxySettings}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.context.annotation.Profile
import org.springframework.stereotype.Component

import scala.concurrent.{Await, Promise}

trait ExecuteRemotelyActorInitializer extends ActorInitializer with Logging {
  def actorProps(): Props = ExecuteRemotelyActor.props
}

@Component
class ExecuteRemotelyActorHolder @Autowired()(xlrConfig: XlrConfig) {

  private val actorPromise = Promise[ActorRef]()

  def actorRef(): ActorRef = {
    Await.result(actorPromise.future, xlrConfig.timeouts.systemInitialization)
  }

  def resolveActorRef(actorRef: ActorRef): Unit = actorPromise.success(actorRef)
}

@Component
@Profile(Array(STANDALONE, HOT_STANDBY))
class NonClusteredExecuteRemotelyActorInitializer @Autowired()(systemHolder: ActorSystemHolder,
                                                               executeRemotelyActorHolder: ExecuteRemotelyActorHolder) extends ExecuteRemotelyActorInitializer {

  lazy val executeRemotelyActor: ActorRef = systemHolder.actorSystem.actorOf(actorProps())

  override def initialize(): Unit = {
    logger.debug("Initializing non-clustered execute remotely actor...")
    executeRemotelyActorHolder.resolveActorRef(executeRemotelyActor)
    executeRemotelyActor
  }
}

@Component
@Profile(Array(FULL))
class ClusteredExecuteRemotelyActorInitializer @Autowired()(systemHolder: ActorSystemHolder,
                                                            executeRemotelyActorHolder: ExecuteRemotelyActorHolder) extends ExecuteRemotelyActorInitializer {

  lazy val executeRemotelyActor: ActorRef = clusteredExecuteRemotelyActor(actorProps())

  override def initialize(): Unit = {
    logger.debug("Initializing clustered execute remotely actor...")
    executeRemotelyActorHolder.resolveActorRef(executeRemotelyActor)
    executeRemotelyActor
  }

  private def clusteredExecuteRemotelyActor(actorProps: Props): ActorRef = {
    systemHolder.actorSystem.actorOf(ClusterSingletonManager.props(
      singletonProps = actorProps,
      terminationMessage = PoisonPill,
      settings = ClusterSingletonManagerSettings(systemHolder.actorSystem)),
      name = ExecuteRemotelyActor.name)

    systemHolder.actorSystem.actorOf(ClusterSingletonProxy.props(
      singletonManagerPath = s"user/${ExecuteRemotelyActor.name}",
      settings = ClusterSingletonProxySettings(systemHolder.actorSystem)),
      name = s"${ExecuteRemotelyActor.name}Proxy")
  }
}
