package com.xebialabs.xlrelease.runner.impl

import com.xebialabs.xlrelease.runner.domain._
import com.xebialabs.xlrelease.runner.impl.RunnerControlChannelActor.{CloseControlChannel, OpenControlChannel, SendCommand}
import com.xebialabs.xlrelease.runner.impl.RunnerProxyActor.{AddChannel, RemoveChannel}
import com.xebialabs.xlrelease.support.pekko.spring.SpringActor
import com.xebialabs.xlrelease.support.serialization.SerializableMsg
import org.apache.pekko.actor.{Actor, ActorLogging, Cancellable, NoSerializationVerificationNeeded}

import java.util.UUID
import jakarta.ws.rs.core.MediaType
import jakarta.ws.rs.sse.{OutboundSseEvent, Sse, SseEventSink}
import scala.concurrent.duration.DurationInt

@SpringActor
class RunnerControlChannelActor(runnerProxyFactory: RunnerProxyFactory) extends Actor with ActorLogging {

  override def receive: Receive = onMessage(None, None, Map.empty)

  private def onMessage(sink: Option[SseEventSink], sse: Option[Sse], pingCommandSchedulers: Map[RunnerId, Cancellable]): Receive = {
    case OpenControlChannel(runnerId, newSink, newSse) =>
      // no need to persist
      sink.foreach(_.close())
      pingCommandSchedulers.get(runnerId).foreach(_.cancel())
      // schedule a ping command for the opened control channel
      val newPingCommandScheduler = getPingCommandScheduler(runnerId)
      context.become(onMessage(Option(newSink), Option(newSse), pingCommandSchedulers + (runnerId -> newPingCommandScheduler)))
      // subscribe this channel as command channel to RunnerProxyActor
      runnerProxyFactory.create(()) ! AddChannel(runnerId, controlChannel = self)
    case CloseControlChannel(runnerId) =>
      // no need to persist
      sink.foreach(_.close())
      pingCommandSchedulers.get(runnerId).foreach(_.cancel())
      context.become(onMessage(None, None, pingCommandSchedulers - runnerId))
      // unsubscribe channel
      runnerProxyFactory.create(()) ! RemoveChannel(runnerId, controlChannel = self)
    case SendCommand(runnerId, command: PersistedCommand) =>
      // cancel the existing scheduler if the new command is sent to the remote runner
      // and create a new one
      pingCommandSchedulers.get(runnerId).foreach(_.cancel())
      sink.filterNot(_.isClosed).foreach(_.send(createSseEvent(command, sse.map(_.newEventBuilder()).get)))
      val newPingCommandScheduler = getPingCommandScheduler(runnerId)
      context.become(onMessage(sink, sse, pingCommandSchedulers + (runnerId -> newPingCommandScheduler)))
  }

  private def getPingCommandScheduler(runnerId: RunnerId): Cancellable = {
    val commandId = UUID.randomUUID().toString
    val command = PersistedCommand(commandId, Ping())
    context.system.scheduler.scheduleOnce(30.seconds, context.parent, SendCommand(runnerId, command))(context.system.dispatcher)
  }

  private def createSseEvent(persistedCommand: PersistedCommand, eventBuilder: OutboundSseEvent.Builder): OutboundSseEvent = {
    val command = persistedCommand.command
    val eventName = command.getClass.getSimpleName
    eventBuilder.name(eventName)
      .mediaType(MediaType.APPLICATION_JSON_TYPE)
      .id(persistedCommand.commandId)
      .data(persistedCommand)
      .build()
  }
}

object RunnerControlChannelActor {
  def actorName(runnerId: RunnerId): String = {
    s"control-channel-${runnerId.shortId()}"
  }

  sealed trait ChannelCommand extends SerializableMsg {
    def runnerId: RunnerId
  }

  case class OpenControlChannel(runnerId: RunnerId, newSink: SseEventSink, newSse: Sse) extends ChannelCommand with NoSerializationVerificationNeeded

  case class CloseControlChannel(runnerId: RunnerId) extends ChannelCommand

  case class SendCommand(runnerId: RunnerId, persistedCommand: PersistedCommand) extends ChannelCommand
}
