package com.xebialabs.xlrelease.runner.impl

import com.xebialabs.deployit.ServerState
import com.xebialabs.deployit.ServerState.Mode
import com.xebialabs.xlrelease.runner.api.v1.JobRunnerResource
import com.xebialabs.xlrelease.runner.api.v1.JobRunnerResource._
import com.xebialabs.xlrelease.runner.domain._
import com.xebialabs.xlrelease.runner.service.RunnerJobService
import com.xebialabs.xlrelease.storage.domain.LogEntry
import grizzled.slf4j.Logging
import org.springframework.stereotype.Controller

import jakarta.ws.rs.sse.{Sse, SseEventSink}
import scala.util.{Failure, Success, Try}


@Controller
class JobRunnerResourceImpl(runnerJobService: RunnerJobService,
                            runnerControlService: RunnerControlService,
                            runnerAvailabilityService: RunnerAvailabilityService)
  extends JobRunnerResource with Logging {

  override def reserveJob(runnerId: String, requestId: String): JobReservationResponse = {
    logger.trace(s"Runner [$runnerId] requested for a job with requestId [$requestId]")
    val possibleJobId = runnerJobService.reserveJob(runnerId)
    JobReservationResponse(requestId, possibleJobId)
  }

  // confirm job for execution (mark it as IN_PROGRESS) and return the data to runner
  override def confirmJob(runnerId: RunnerId, jobId: JobId, requestId: String): ConfirmationResponse = {
    logger.trace(s"Runner [$runnerId] request to confirm job $jobId received with requestId [$requestId]")
    val isConfirmed = runnerJobService.confirmJob(runnerId, jobId)
    ConfirmationResponse(requestId, isConfirmed)
  }


  override def finishJob(runnerId: String, requestId: String, jobResult: JobResult): ConfirmationResponse = {
    logger.trace(s"Received result for jobId [${jobResult.getJobIdentifier}] from runner [$runnerId] with requestId [$requestId]")
    runnerJobService.finishJob(jobResult)
    ConfirmationResponse(requestId, isConfirmed = true)
  }

  override def log(runnerId: String, requestId: String, logEntryRequest: LogEntryRequest): ConfirmationResponse = {
    logger.trace(s"Received log entry [$logEntryRequest] from runner [$runnerId] with requestId [$requestId]")
    val logEntry: LogEntry = LogEntry(logEntryRequest.taskId,
      logEntryRequest.executionId,
      logEntryRequest.jobId,
      logEntryRequest.chunk,
      logEntryRequest.lastEntryTimestamp,
      logEntryRequest.payload,
      logEntryRequest.uriScheme
    )
    Try(runnerJobService.log(logEntry)) match {
      case Success(_) => ConfirmationResponse(requestId, isConfirmed = true)
      case Failure(exception) =>
        logger.warn(s"Error occurred while processing log entry [$runnerId] with requestId [$requestId] " +
          s"- sending back not-confirmed; error was: ${exception.getMessage}")
        ConfirmationResponse(requestId, isConfirmed = false)
    }

  }

  override def executeDirectives(runnerId: String, requestId: String, directives: Seq[JobDirective]): ConfirmationResponse = {
    logger.trace(s"Received directives from runner [$runnerId] with requestId [$requestId]")
    runnerJobService.executeDirectives(directives)
    ConfirmationResponse(requestId, isConfirmed = true)
  }

  override def commands(runnerId: RunnerId, sink: SseEventSink, sse: Sse): Unit = {
    logger.trace(s"Received request to open control channel for runnerId [$runnerId]")
    runnerControlService.watch(runnerId, sink, sse)
  }

  override def confirmCommand(runnerId: String, commandId: String): ConfirmationResponse = {
    logger.trace(s"Received confirmation for commandId [$commandId] from runnerId [$runnerId]")
    runnerControlService.confirmCommand(runnerId, commandId)
    ConfirmationResponse(commandId, isConfirmed = true)
  }

  override def health(runnerId: String, requestId: String): JobRunnerResource.HealthResponse = {
    logger.trace(s"Received request for health check from runnerId [$runnerId] with requestId [$requestId]")
    runnerControlService.alive(runnerId, System.currentTimeMillis())
    runnerAvailabilityService.alive(runnerId)
    val serverState = ServerState.getInstance().getCurrentMode match {
      case Mode.STARTING => ReleaseServerStates.STARTING
      case Mode.RUNNING => ReleaseServerStates.RUNNING
      case Mode.MAINTENANCE => ReleaseServerStates.MAINTENANCE
    }
    HealthResponse(requestId, serverState)
  }
}
