package com.xebialabs.xlrelease.runner.impl.security.authentication

import com.xebialabs.deployit.security.authentication.AuthenticationFailureException
import com.xebialabs.xlrelease.authentication.InvalidTokenException
import com.xebialabs.xlrelease.runner.service.RunnerTokenService
import com.xebialabs.xlrelease.security.authentication.{RunnerAuthenticationToken, TokenExpiredException}
import com.xebialabs.xlrelease.utils.TokenGenerator
import grizzled.slf4j.Logging
import org.springframework.security.authentication.AuthenticationProvider
import org.springframework.security.core.Authentication
import org.springframework.stereotype.Component

import java.util.Collections

@Component
class RunnerAuthenticationProvider(runnerTokenService: RunnerTokenService) extends AuthenticationProvider with Logging {

  override def supports(authentication: Class[_]): Boolean = authentication.isAssignableFrom(classOf[RunnerAuthenticationToken])

  override def authenticate(authentication: Authentication): Authentication = {
    val bearer = authentication.asInstanceOf[RunnerAuthenticationToken]
    try {
      val token = bearer.getCredentials.toString
      val tokenHash = TokenGenerator.hash(token)
      val runnerId = runnerTokenService.findByToken(tokenHash).getOrElse(
        throw new AuthenticationFailureException("Cannot authenticate with supplied runner authentication token"))
      if (runnerTokenService.isExpired(tokenHash)) {
        throw TokenExpiredException(s"The supplied runner authentication token has expired")
      }
      new RunnerAuthenticationToken(runnerId, token, Collections.emptyList)
    } catch {
      case ex: AuthenticationFailureException => throw InvalidTokenException(ex.getMessage)
    }
  }

}
