import { all, call, CallEffect, put, select, takeEvery } from 'redux-saga/effects';
import { IHttpResponse } from 'angular';
import { remoteRunners, UpdateRemoteRunnerStatePayload } from './remote-runners.reducer';
import { RemoteRunner } from '../remote-runners.types';
import { PayloadAction } from '@reduxjs/toolkit';
import { getRemoteRunnersSelector } from './remote-runners.selectors';
import { ConfigurationCallbackPayload } from '../../../../../../../../../core/xlr-ui/app/features/configuration/types';
import { httpDELETE, httpGET, httpPUT } from '../../../../../../../../../core/xlr-ui/app/features/common/services/http';

const { setIsLoading, loadRemoteRunners, setRemoteRunners, reloadRemoteRunners, updateRemoteRunnerState, deleteRemoteRunner } = remoteRunners.actions;
const url = 'api/v1';

export function* withLoadingState<R>(effect: CallEffect) {
    try {
        yield put(setIsLoading(true));
        const result: R = yield effect;
        return result;
    } finally {
        yield put(setIsLoading(false));
    }
}

export function* loadRemoteRunnersAction() {
    const { data }: IHttpResponse<RemoteRunner[]> = yield call(
        withLoadingState,
        call(httpGET, `${url}/config/byTypeAndTitle?configurationType=xlrelease.RemoteJobRunner`),
    );
    yield put(setRemoteRunners(data));
}

export function* reloadRemoteRunnersAction(_action: PayloadAction<ConfigurationCallbackPayload>) {
    yield call(loadRemoteRunnersAction);
}

export function* updateRemoteRunnerStateAction(action: PayloadAction<UpdateRemoteRunnerStatePayload>) {
    const remoteRunnerId = action.payload.id;
    const enable = action.payload.enable;
    const { data }: IHttpResponse<RemoteRunner> = yield call(httpGET, `${url}/config/${remoteRunnerId}`);
    const updateRemoteRunnerBody = {
        ...data,
        enabled: enable,
    };
    const { data: updatedRunner }: IHttpResponse<RemoteRunner> = yield call(httpPUT, `${url}/config/${updateRemoteRunnerBody.id}`, updateRemoteRunnerBody);
    const runners: RemoteRunner[] = yield select(getRemoteRunnersSelector);
    const updatedRunners: RemoteRunner[] = runners.map((remoteRunner): RemoteRunner => {
        if (remoteRunner.id === updatedRunner.id) return updatedRunner;
        return remoteRunner;
    });
    yield put(setRemoteRunners(updatedRunners));
}

export function* deleteRemoteRunnerAction(action: PayloadAction<RemoteRunner>) {
    const remoteRunnerToDelete = action.payload;
    yield call(httpDELETE, `${url}/config/${remoteRunnerToDelete.id}`);
    const runners: RemoteRunner[] = yield select(getRemoteRunnersSelector);
    yield put(setRemoteRunners([...runners].filter((runner) => runner.id !== remoteRunnerToDelete.id)));
}

export default function* remoteRunnersSaga() {
    yield all([
        takeEvery(loadRemoteRunners, loadRemoteRunnersAction),
        takeEvery(reloadRemoteRunners, reloadRemoteRunnersAction),
        takeEvery(updateRemoteRunnerState, updateRemoteRunnerStateAction),
        takeEvery(deleteRemoteRunner, deleteRemoteRunnerAction),
    ]);
}
