import React, { useState } from 'react';
import moment from 'moment';
import { DotChip, DotColumnHeader, DotIcon, DotSwitch, DotTable, DotTableActions, DotTypography, TableRowProps } from '@digital-ai/dot-components';
import './remote-runners-table.less';
import { remoteRunners, UpdateRemoteRunnerStatePayload } from '../ducks/remote-runners.reducer';
import { RemoteRunner } from '../remote-runners.types';
import angular from 'angular';
import { useAppDispatch } from '../../../../../../../../../core/xlr-ui/app/js/hooks';
import { ChipGroup } from '../../../../../../../../../core/xlr-ui/app/react/components/chip-group/chip-group.component';
import { Search } from '../../../../../../../../../core/xlr-ui/app/react/components/search/search';

export const headerDefinitions: DotColumnHeader[] = [
    { id: 'status', label: 'Status' },
    { id: 'name', label: 'Name' },
    { id: 'version', label: 'Version' },
    { id: 'capabilities', label: 'Capabilities' },
    { id: 'lastActivity', label: 'Last activity' },
    { id: 'state', label: 'State' },
    { id: 'actions', align: 'right', label: 'Actions', sortable: false },
];

export interface RemoteRunnersTableProps {
    isLoading: boolean;
    onDeleteClick: (runner: RemoteRunner) => void;
    onEditClick: (runner: RemoteRunner) => void;
    runners: RemoteRunner[];
}

const { updateRemoteRunnerState } = remoteRunners.actions;

export const RemoteRunnersTable = ({ isLoading, onDeleteClick, onEditClick, runners }: RemoteRunnersTableProps) => {
    const dispatch = useAppDispatch();

    const [searchText, setSearchText] = useState<string>('');

    const getRunnerStatus = (remoteRunner: RemoteRunner) => {
        return (
            <DotChip
                className={remoteRunner.available ? 'status-success' : 'status-error'}
                data-testid={`status-chip-${remoteRunner.id}`}
                error={!remoteRunner.available}
                isDeletable={false}
                size="small"
                startIcon={<DotIcon className={remoteRunner.available ? 'success' : 'error'} iconId={remoteRunner.available ? 'check-solid' : 'error-solid'} />}
            >
                {remoteRunner.available ? 'Running' : 'Disconnected'}
            </DotChip>
        );
    };

    const getRunnerCapabilities = (remoteRunner: RemoteRunner) => {
        return <ChipGroup labels={remoteRunner.capabilities} />;
    };

    const getRunnerState = (remoteRunner: RemoteRunner) => {
        return (
            <DotSwitch
                checked={remoteRunner.enabled}
                data-testid={`state-switch-${remoteRunner.id}`}
                key={remoteRunner.id}
                label={remoteRunner.enabled ? 'Disable' : 'Enable'}
                onChange={(e) => handleStateChange(remoteRunner, e.target.checked)}
            />
        );
    };

    const getRunnerActions = (remoteRunner: RemoteRunner) => {
        return (
            <DotTableActions
                actions={[
                    { iconId: 'edit', id: 'edit', label: 'Edit', onClick: () => onEditClick(remoteRunner), tooltip: 'Edit' },
                    { iconId: 'delete', id: 'delete', label: 'Delete', onClick: () => onDeleteClick(remoteRunner), tooltip: 'Delete' },
                ]}
                id={remoteRunner.id}
            />
        );
    };

    const handleStateChange = (remoteRunner: RemoteRunner, enable: boolean) => {
        const payload: UpdateRemoteRunnerStatePayload = {
            id: remoteRunner.id,
            enable,
        };
        dispatch(updateRemoteRunnerState(payload));
    };

    const filteredRunners: RemoteRunner[] =
        searchText.length > 0 ? runners.filter((remoteRunner) => remoteRunner.runnerName.toLowerCase().includes(searchText.toLowerCase())) : runners;

    const tableData: TableRowProps[] = filteredRunners.map((remoteRunner) => {
        return {
            id: remoteRunner.id,
            rowData: {
                status: getRunnerStatus(remoteRunner),
                name: remoteRunner.runnerName,
                version: remoteRunner.version,
                capabilities: getRunnerCapabilities(remoteRunner),
                lastActivity: remoteRunner.lastActivity ? moment(remoteRunner.lastActivity).fromNow() : 'Never',
                state: getRunnerState(remoteRunner),
                actions: getRunnerActions(remoteRunner),
            },
        };
    });

    return (
        <div className="remote-runners-table">
            <div className="remote-runners-table-title">
                <DotTypography variant="h1">Remote runners</DotTypography>
                <Search
                    defaultValue={searchText}
                    inputPlaceholder="Filter by name"
                    onFilterChange={(e) => setSearchText(e.target.value)}
                    searchClassNames={['mtm', 'mbm']}
                />
            </div>
            <DotTable columns={headerDefinitions} data={tableData} key={angular.toJson(filteredRunners)} loading={isLoading} rowsPerPage={10} />
        </div>
    );
};
