import React, { ChangeEvent } from 'react';
import { ReactWrapper } from 'enzyme';
import { Provider } from 'react-redux';
import { mockResizeObserver, mountWithStoreAndTheme } from '../../../../../../../../../core/xlr-ui/tests/unit/testing-utils';
import { headerDefinitions, RemoteRunnersTable, RemoteRunnersTableProps } from './remote-runners-table';
import { remoteRunners } from '../ducks/remote-runners.reducer';
import { remoteRunnerMock } from '../__mocks__/remote-runners.mocks';
import { Search } from '../../../../../../../../../core/xlr-ui/app/react/components/search/search';
import { DotChip, DotSwitch, DotTable, DotTableActions } from '@digital-ai/dot-components';
import { ChipGroup } from '../../../../../../../../../core/xlr-ui/app/react/components/chip-group/chip-group.component';

const { updateRemoteRunnerState } = remoteRunners.actions;

describe('RemoteRunnersTable component', () => {
    let wrapper: ReactWrapper;
    const dispatch = jest.fn();

    const defaultProps: RemoteRunnersTableProps = {
        isLoading: false,
        onDeleteClick: jest.fn(),
        onEditClick: jest.fn(),
        runners: [],
    };
    const mountComponent = (props: RemoteRunnersTableProps = defaultProps) => {
        wrapper = mountWithStoreAndTheme(<RemoteRunnersTable {...props} />, dispatch);
        expect(wrapper.find(Provider)).toExist();
    };

    const getStatusChip = (id: string) => wrapper.findWhere((node) => node.is(DotChip) && node.props()['data-testid'] === `status-chip-${id}`);
    const getTable = () => wrapper.find(DotTable);

    const getName = (index: number) => getTable().find('tbody tr').at(index).find('td').at(1);

    const getVersion = (index: number) => getTable().find('tbody tr').at(index).find('td').at(2);

    const getLastActivity = (index: number) => getTable().find('tbody tr').at(index).find('td').at(4);

    const getStateSwitch = (id: string) => wrapper.findWhere((node) => node.is(DotSwitch) && node.props()['data-testid'] === `state-switch-${id}`);

    const getTableActions = (id: string) => wrapper.findWhere((node) => node.is(DotTableActions) && node.props().id === id);

    beforeAll(() => {
        mockResizeObserver();
    });

    afterEach(() => {
        wrapper.unmount();
    });

    it('should render empty runners table', () => {
        mountComponent();

        const search = wrapper.find(Search);
        expect(search).toExist();

        const searchProps = search.props();
        expect(searchProps.inputPlaceholder).toBe('Filter by name');

        const table = getTable();
        expect(table).toExist();

        const tableProps = table.props();
        expect(tableProps.columns).toStrictEqual(headerDefinitions);
        expect(tableProps.data.length).toBe(0);
        expect(tableProps.loading).toBe(false);
        expect(tableProps.rowsPerPage).toBe(10);
    });

    it('should render runners table with 1 runner', () => {
        mountComponent({ ...defaultProps, runners: [remoteRunnerMock] });

        const table = getTable();
        expect(table).toExist();

        const tableProps = table.props();
        expect(tableProps.columns).toStrictEqual(headerDefinitions);
        expect(tableProps.data.length).toBe(1);
        expect(tableProps.loading).toBe(false);
        expect(tableProps.rowsPerPage).toBe(10);

        const statusChip = getStatusChip(remoteRunnerMock.id);
        expect(statusChip).toExist();

        const statusChipProps = statusChip.props();
        expect(statusChipProps.className).toBe('status-success');
        expect(statusChipProps.error).toBe(false);
        expect(statusChipProps.isDeletable).toBe(false);
        expect(statusChipProps.children).toBe('Running');

        const nameLabel = getName(0);
        expect(nameLabel.text()).toBe('new-runner');

        const versionLabel = getVersion(0);
        expect(versionLabel.text()).toBe('0.2.0');

        const capabilitiesChips = table.find(ChipGroup);
        expect(capabilitiesChips).toExist();
        expect(capabilitiesChips.props().labels).toStrictEqual(['remote']);

        const lastActivityLabel = getLastActivity(0);
        expect(lastActivityLabel.text()).toBe('Never');

        const stateSwitch = getStateSwitch(remoteRunnerMock.id);
        expect(stateSwitch).toExist();

        const stateSwitchProps = stateSwitch.props();
        expect(stateSwitchProps.checked).toBe(true);
        expect(stateSwitchProps.label).toBe('Disable');

        const tableActions = getTableActions(remoteRunnerMock.id);
        expect(tableActions).toExist();
    });

    it('should render runner with disconnected status', () => {
        mountComponent({ ...defaultProps, runners: [{ ...remoteRunnerMock, available: false }] });

        const statusChip = getStatusChip(remoteRunnerMock.id);
        expect(statusChip).toExist();

        const statusChipProps = statusChip.props();
        expect(statusChipProps.className).toBe('status-error');
        expect(statusChipProps.error).toBe(true);
        expect(statusChipProps.isDeletable).toBe(false);
        expect(statusChipProps.children).toBe('Disconnected');
    });

    it('should filter table data when search input is filled', () => {
        mountComponent({ ...defaultProps, runners: [remoteRunnerMock] });

        let table = getTable();
        expect(table).toExist();
        expect(table.props().data.length).toBe(1);

        const search = wrapper.find(Search);
        expect(search).toExist();

        const e = { target: { value: 'non matching criteria' } } as unknown as ChangeEvent<HTMLInputElement>;
        search.invoke('onFilterChange')?.(e);
        wrapper.update();

        table = getTable();
        expect(table).toExist();
        expect(table.props().data.length).toBe(0);
    });

    it('should run update runner saga when state switch is clicked', () => {
        mountComponent({ ...defaultProps, runners: [remoteRunnerMock] });

        const table = getTable();
        expect(table).toExist();
        expect(table.props().data.length).toBe(1);

        const stateSwitch = wrapper.find(DotSwitch);
        expect(stateSwitch).toExist();

        stateSwitch.invoke('onChange')?.({ target: { checked: false } } as never);
        expect(dispatch).toHaveBeenCalledWith(updateRemoteRunnerState({ id: remoteRunnerMock.id, enable: false }));
    });
});
