package com.xebialabs.xlrelease.runner.impl

import com.typesafe.config.Config
import com.xebialabs.xlplatform.cluster.ClusterMode
import com.xebialabs.xlrelease.actors.ActorSystemHolder
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.runner.domain._
import com.xebialabs.xlrelease.runner.impl.RunnerProxyActor.{RunnerHealth, RunnerProxyCommand, actorName}
import com.xebialabs.xlrelease.support.pekko.spring.SpringExtension
import org.apache.pekko.actor.ActorRef
import org.apache.pekko.cluster.sharding.{ClusterSharding, ClusterShardingSettings, ShardRegion}
import org.springframework.context.annotation.Profile
import org.springframework.stereotype.Component

import javax.annotation.PostConstruct
import scala.concurrent.duration.DurationInt
import scala.concurrent.{Await, Future}
import scala.util.{Failure, Success, Try}

sealed trait RunnerProxyFactory extends ActorFactory[Unit]

@Profile(Array(ClusterMode.STANDALONE))
@Component
class NonClusteredRunnerProxyFactory(springExtension: SpringExtension) extends RunnerProxyFactory {

  private def runnerProxyActorSupervisor(): ActorRef = {
    val supervisorPath = "user/runner-proxy-supervisor"
    val timeout = 5.seconds
    val f: Future[ActorRef] = springExtension.getActorSystem.actorSelection(supervisorPath).resolveOne(timeout)
    Try(Await.result(f, timeout)) match {
      case Failure(_) => springExtension.actorOf(classOf[RunnerProxyActorSupervisor], "runner-proxy-supervisor")
      case Success(ref) => ref
    }
  }

  override def create(params: Unit): ActorRef = runnerProxyActorSupervisor()

}

@Profile(Array(ClusterMode.FULL))
@Component
class ClusteredRunnerProxyFactory(xlrConfig: XlrConfig, systemHolder: ActorSystemHolder, springExtension: SpringExtension) extends RunnerProxyFactory {

  private def actorSystem = systemHolder.actorSystem

  @PostConstruct
  def init(): Unit = {
    shardRegion
  }

  lazy val shardRegion: ActorRef = {
    val sharding = ClusterSharding(actorSystem)
    val originalConfig = actorSystem.settings.config.getConfig("pekko.cluster.sharding")
    val jobRunnerShardingConfig: Config = xlrConfig.xl.getConfig("job-runner.pekko.cluster.sharding").withFallback(originalConfig)
    val shardingSettings = ClusterShardingSettings(jobRunnerShardingConfig)

    val actorProps = springExtension.props(classOf[RunnerProxyActor])
    val shardRegion = sharding.start(
      typeName = RunnerProxyActor.SHARDING_TYPE_NAME,
      entityProps = actorProps,
      settings = shardingSettings,
      extractEntityId = extractEntityId,
      extractShardId = extractShardId
    )
    shardRegion
  }

  override def create(params: Unit): ActorRef = shardRegion

  private def extractEntityId: ShardRegion.ExtractEntityId = {
    case health: RunnerHealth =>

      val entityId = actorName(health.runnerId)
      (entityId, health)
    case cmd: RunnerProxyCommand =>
      val entityId = actorName(cmd.runnerId)
      (entityId, cmd)
  }

  private def extractShardId: ShardRegion.ExtractShardId = {
    case health: RunnerHealth =>
      val entityId = actorName(health.runnerId)
      entityId.shardId()
    case cmd: RunnerProxyCommand =>
      val entityId = actorName(cmd.runnerId)
      entityId.shardId()
    case ShardRegion.StartEntity(entityId) =>
      entityId.shardId()
  }
}
