package com.xebialabs.xlrelease.runner.impl

import com.xebialabs.xlrelease.domain.runner.{JobRunner, JobRunnerOps, RemoteJobRunner}
import RunnerProxyActor.NewCommand
import com.fasterxml.jackson.databind.ObjectMapper
import com.fasterxml.jackson.module.scala.DefaultScalaModule
import com.xebialabs.xlrelease.domain.RemoteExecution
import com.xebialabs.xlrelease.runner.domain._
import com.xebialabs.xlrelease.service.ConfigurationService
import com.xebialabs.xlrelease.support.akka.spring.ScalaSpringAwareBean
import grizzled.slf4j.Logging

trait RemoteJobRunnerOps extends JobRunnerOps

object RemoteJobRunnerOps extends RemoteJobRunnerOps with ScalaSpringAwareBean with ContainerTaskToScriptJobDataConversion with Logging {

  private lazy val remoteRunnerProxyActorFactory = springBean[RunnerProxyFactory]
  override lazy val configurationService: ConfigurationService = springBean[ConfigurationService]

  override def objectMapper: ObjectMapper = {
    val mapper = new ObjectMapper()
    mapper.registerModule(DefaultScalaModule)
    mapper
  }

  override def start(runner: JobRunner): Unit = {
    logger.debug(s"Start invoked on $runner")
    val command = EnableRunner(runnerCapacity = runner.capacity, encryptJobData = runner.isEncryptJobData())
    remoteRunnerProxyActorFactory.create(()) ! NewCommand(runner.getId, command)
  }

  override def stop(runner: JobRunner): Unit = {
    logger.debug(s"Stop invoked on $runner")
    val command = DisableRunner()
    remoteRunnerProxyActorFactory.create(()) ! NewCommand(runner.getId, command)
  }

  override def delete(runner: JobRunner): Unit = {
    logger.debug(s"Delete invoked on $runner")
    val command = DeleteRunner()
    remoteRunnerProxyActorFactory.create(()) ! NewCommand(runner.getId, command)
  }

  override def abortJob(runner: JobRunner, jobId: Long): Unit = {
    logger.debug(s"Abort job invoked on $runner")
    val command = Abort(jobId)
    remoteRunnerProxyActorFactory.create(()) ! NewCommand(runner.getId, command)
  }

  override def accepts(runner: JobRunner): Boolean = {
    runner.isInstanceOf[RemoteJobRunner]
  }

  override def execute(runner: JobRunner, remoteExecution: RemoteExecution): String = {
    logger.debug(s"Execute job invoked on $runner")


    val scriptJobData = toScriptJobData(runner.getId, remoteExecution) match {
      case Some(value) => value
      case None =>
        throw ContainerScriptException(s"Unable to create script $remoteExecution job data")
    }

    val command = ExecuteScript(scriptJobData)
    remoteRunnerProxyActorFactory.create(()) ! NewCommand(runner.getId, command)

    scriptJobData.executionId
  }
}
