package com.xebialabs.xlrelease.runner.impl

import akka.actor.ActorRef
import akka.pattern.ask
import akka.util.Timeout
import com.fasterxml.jackson.core.`type`.TypeReference
import com.fasterxml.jackson.databind.ObjectMapper
import com.fasterxml.jackson.module.scala.DefaultScalaModule
import com.xebialabs.xlrelease.builder.TaskBuilder.newRemoteExecution
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.domain.RemoteExecution
import com.xebialabs.xlrelease.domain.runner.JobRunner
import com.xebialabs.xlrelease.support.akka.spring.SpringExtension
import org.springframework.stereotype.Service

import java.util
import scala.concurrent.Await
import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

trait RunnerScriptService {
  def executeScript[T](script: String, inputParameters: util.Map[String, AnyRef], typeReference: TypeReference[T]): T
  def scriptExecuted(scriptExecutionId: String, payload: util.Map[String, AnyRef]): Unit
}

@Service
class RunnerScriptServiceImpl(springExtension: SpringExtension,
                              xlrConfig: XlrConfig,
                              jobRunnerService: JobRunnerService) extends RunnerScriptService {
  private lazy val actorRef: ActorRef = springExtension.actorOf(classOf[ExecuteRemotelyActor])

  private val CommandResponseKey = "commandResponse"
  private val ErrorMessageKey = "errorMessage"

  private def objectMapper: ObjectMapper = {
    val mapper = new ObjectMapper()
    mapper.registerModule(DefaultScalaModule)
    mapper
  }

  private def askAndAwait[T](jobRunner: JobRunner, remoteExecution: RemoteExecution): T = {
    implicit val askTimeout: Timeout = xlrConfig.timeouts.releaseActionResponse
    try {
      Await.result(
        actorRef ? ExecuteRemotely(jobRunner, remoteExecution),
        askTimeout.duration
      ).asInstanceOf[T]
    } catch {
      case e: Exception =>
        throw ContainerScriptException(e.getMessage)
    }
  }

  override def executeScript[T](script: String, inputParameters: util.Map[String, AnyRef], typeReference: TypeReference[T]): T = {
    val scriptResponse = executeRemotely(script, inputParameters).payload.asScala.toMap

    scriptResponse.get(CommandResponseKey) match {
      case Some(commandResponse) =>
        Try(objectMapper.readValue(objectMapper.writeValueAsString(commandResponse), typeReference)) match {
          case Success(result) => result
          case Failure(exception) => throw ContainerScriptException(s"Error deserializing container $script response: $exception")
        }
      case None if scriptResponse.contains(ErrorMessageKey) =>
        val error = objectMapper.readValue(objectMapper.writeValueAsString(scriptResponse(ErrorMessageKey)), classOf[String])
        throw ContainerScriptException(s"Container script $script failed with error: $error")
      case _ =>
        throw ContainerScriptException(s"Failed to process container script $script result")
    }
  }
  private def executeRemotely(script: String, inputParameters: util.Map[String, AnyRef]): ExecuteRemotelyResponse = {
    val scriptTask = newRemoteExecution(script)
      .withInputParameters(inputParameters)
      .build()

    val runner = jobRunnerService.findScriptExecutor() match {
      case Some(runner) => runner
      case None =>
        throw ContainerScriptException(s"Cannot find active remote runner for executing script '$script'")
    }

    askAndAwait[ExecuteRemotelyResponse](runner, scriptTask)
  }

  override def scriptExecuted(scriptExecutionId: String, payload: util.Map[String, AnyRef]): Unit = {
    actorRef ! ExecuteRemotelyResponse(scriptExecutionId, payload)
  }
}

case class ContainerScriptException(msg: String) extends Exception(msg)
