package com.xebialabs.xlrelease.runner.impl

import com.xebialabs.deployit.ServerState
import com.xebialabs.deployit.ServerState.Mode
import com.xebialabs.xlrelease.runner.api.internal.JobRunnerResource
import com.xebialabs.xlrelease.runner.api.internal.JobRunnerResource.{HealthResponse, ReleaseServerStates}
import com.xebialabs.xlrelease.runner.domain._
import com.xebialabs.xlrelease.runner.service.RunnerJobService
import com.xebialabs.xlrelease.storage.domain.LogEntry
import grizzled.slf4j.Logging
import org.springframework.stereotype.Controller

import javax.ws.rs.sse.{Sse, SseEventSink}

@Controller
class JobRunnerResourceImpl(runnerJobService: RunnerJobService,
                            runnerControlService: RunnerControlService)
  extends JobRunnerResource with Logging {

  // TODO any of these messages can fail or timeout - handle errors on the client side
  //  i.e. see how we can get and log meaningful data
  //  remember we cannot use debugger and resteasy hides interesting details (like when deserialization fails)

  override def getJob(runnerId: String): JobData = {
    logger.trace(s"Runner [$runnerId] requested for a job")
    val jobData = runnerJobService.getJob(runnerId)
    jobData
  }

  override def finishJob(jobResult: JobResult): Unit = {
    logger.trace(s"Received result for jobId [${jobResult.jobId}] from runner [${jobResult.runnerId}]")
    runnerJobService.finishJob(jobResult)
  }

  override def log(logEntry: LogEntry): Unit = {
    logger.trace(s"Received log entry [$logEntry]")
    runnerJobService.log(logEntry)
  }

  override def executeDirectives(directives: Seq[JobDirective]): Unit = {
    runnerJobService.executeDirectives(directives)
  }

  override def commands(runnerId: RunnerId, sink: SseEventSink, sse: Sse): Unit = {
    logger.trace(s"Received request to open control channel for runnerId [$runnerId]")
    runnerControlService.watch(runnerId, sink, sse)
  }

  override def confirmCommand(runnerId: String, commandId: String): Unit = {
    logger.trace(s"Received confirmation for commandId [$commandId] from runnerId [$runnerId]")
    runnerControlService.confirmCommand(runnerId, commandId)
  }

  override def health(): JobRunnerResource.HealthResponse = {
    val serverState = ServerState.getInstance().getCurrentMode match {
      case Mode.STARTING => ReleaseServerStates.STARTING
      case Mode.RUNNING => ReleaseServerStates.RUNNING
      case Mode.MAINTENANCE => ReleaseServerStates.MAINTENANCE
    }
    HealthResponse(serverState)
  }
}
