package com.xebialabs.xlrelease.runner.impl

import com.fasterxml.jackson.databind.ObjectMapper
import com.fasterxml.jackson.databind.annotation.JsonDeserialize
import com.fasterxml.jackson.databind.deser.std.UntypedObjectDeserializer
import com.fasterxml.jackson.module.scala.DefaultScalaModule
import com.xebialabs.deployit.plugin.api.reflect.{PropertyDescriptor, PropertyKind, Type}
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.xlrelease.actors.ReleaseActorService
import com.xebialabs.xlrelease.domain.facet.TaskReportingRecord
import com.xebialabs.xlrelease.domain.runner.RemoteJobRunner
import com.xebialabs.xlrelease.domain.utils.ReleaseCloneHelper
import com.xebialabs.xlrelease.domain.{Configuration, ContainerTask, Task}
import com.xebialabs.xlrelease.repository.{IdType, TaskRepository}
import com.xebialabs.xlrelease.runner.domain._
import com.xebialabs.xlrelease.runner.impl.RunnerJobServiceImpl._
import com.xebialabs.xlrelease.runner.service.RunnerJobService
import com.xebialabs.xlrelease.scheduler.{ContainerTaskJob, FailJob, JobQueue, StopWorkerThread}
import com.xebialabs.xlrelease.script.{EncryptionHelper, FailureContainerTaskResult, SuccessContainerTaskResult}
import com.xebialabs.xlrelease.serialization.json.jackson.CiDeserializer
import com.xebialabs.xlrelease.serialization.json.repository.ResolveOptions
import com.xebialabs.xlrelease.service.{CommentService, ConfigurationService, ConfigurationVariableService, FacetService}
import com.xebialabs.xlrelease.storage.domain.LogEntry
import com.xebialabs.xlrelease.support.akka.SerializableMsg
import grizzled.slf4j.Logging
import org.apache.commons.lang3.exception.ExceptionUtils
import org.springframework.stereotype.Component

import java.util
import scala.annotation.meta.field
import scala.jdk.CollectionConverters._
import scala.util.Try

@Component
class RunnerJobServiceImpl(jobQueue: JobQueue,
                           releaseActorService: ReleaseActorService,
                           configurationVariableService: ConfigurationVariableService,
                           facetService: FacetService,
                           commentService: CommentService,
                           taskRepository: TaskRepository,
                           val configurationService: ConfigurationService)
  extends RunnerJobService
    with InputContextFactory
    with InputContextSerialization
    with OutputContextDeserialization
    with Logging {

  private lazy val _objectMapper: ObjectMapper = {
    val mapper = new ObjectMapper()
    mapper.registerModule(DefaultScalaModule)
    mapper
  }

  override def objectMapper: ObjectMapper = _objectMapper

  override def getJob(runnerId: RunnerId): JobData = {
    import scala.jdk.CollectionConverters._
    jobQueue.get(runnerId) match {
      case Some(job: ContainerTaskJob) =>
        val task: ContainerTask = job.taskRef.get()
        // resolve variables
        val globalVariables = task.getRelease.getGlobalVariables
        configurationVariableService.resolveFromCi[ContainerTask](task, globalVariables)(ci => ci.getInputProperties.asScala)
        // decrypt task
        // TODO check if clone is really needed as we suspect task is already clones by ExecutionService
        val containerTask = ReleaseCloneHelper.clone(task)
        EncryptionHelper.decrypt(containerTask.getRelease)
        EncryptionHelper.decrypt(containerTask)
        val inputContext = create(containerTask)
        val inputContextData: ContextData = serialize(inputContext, runnerId)
        val abortTimeout = containerTask.getAbortTimeout()
        val maxRetryAttempts = containerTask.getMaxRetryAttempts()
        val retryDelay = containerTask.getRetryDelay()
        val taskImg = containerTask.getProperty[String]("image")
        val registryUrl = containerTask.getProperty[String]("registryUrl")
        ContainerJobData(job.id, job.taskId, taskImg, abortTimeout, maxRetryAttempts, retryDelay, inputContextData, registryUrl)
      case Some(failJob: FailJob) =>
        FailJobData(failJob.job.id)
      case Some(StopWorkerThread()) =>
        StopWorkerJobData()
      case Some(job) =>
        SomeJobData(job.id)
      case None => NoJobData()
    }
  }

  override def failJob(jobId: JobId): Unit = {
    jobQueue.finish(jobId)
  }

  override def finishJob(jobResult: JobResult): Unit = {
    jobQueue.finish(jobResult.jobId)
    jobResult match {
      case result: TaskJobResult =>
        val taskResult = result match {
          case TaskJobSuccess(taskId, _, runnerId, containerStatusCode, outputContextData) =>
            try {
              val outputContext = deserialize(runnerId, outputContextData)
              if (containerStatusCode == 0 && outputContext.exitCode == 0) {
                // TODO here a lot can go "wrong": reportingRecords can be null, outputProperties can be null, etc.
                if (null != outputContext.reportingRecords) {
                  Try(outputContext.reportingRecords.asScala.foreach { record =>
                    val task: Task = taskRepository.findById(record.getTargetId, ResolveOptions.WITHOUT_DECORATORS)
                    record.setCreatedViaApi(true)
                    record.setRetryAttemptNumber(task.getFailuresCount)

                    facetService.addTaskReportingRecord(record, applyTaskAttributes = true)
                  }).recover {
                    case t: Throwable => logger.error(s"Unable to add task reporting record for task [$taskId]", t)
                  }
                }
                SuccessContainerTaskResult(taskId, outputContext.outputProperties)
              } else {
                FailureContainerTaskResult(taskId, s"Container exited with status code: $containerStatusCode")
              }
            } catch {
              case ex: Exception =>
                FailureContainerTaskResult(taskId, ExceptionUtils.getStackTrace(ex))
            }
          case TaskJobFailure(taskId, _, _, ex) =>
            FailureContainerTaskResult(taskId, ex)
        }
        releaseActorService.finishContainerTask(taskResult)
      case jobResult =>
        logger.warn(s"Can't handle supplied job result of type [${jobResult.getClass.getSimpleName}]")
    }
  }

  override def log(logEntry: LogEntry): Unit = {
    if (logEntry.chunk == 1) {
      // add comment to the task that points to the log service endpoint
      // (e.g. endpoint that streams content as series of SSE)
      val taskId = logEntry.taskId
      val jobId = logEntry.jobId
      val viewTaskId = IdType.DOMAIN.convertToViewId(taskId)
      val url = s"/tasks/$viewTaskId/job/$jobId/log"
      val followUrl = s"$url/follow"
      val followFromStartUrl = s"$followUrl?fromStart=true"
      val comment = s"- Follow logs [here]($followUrl) \n" +
        s"- Follow logs from start [here]($followFromStartUrl) \n" +
        s"- Download logs from [here]($url)"
      createComment(taskId, comment)
    }
  }

  private def createComment(taskId: String, comment: String): Unit = {
    val task: Task = taskRepository.findById(taskId, ResolveOptions.WITHOUT_DECORATORS)
    commentService.appendComment(task, null, comment)
  }

  override def executeDirectives(directives: Seq[JobDirective]): Unit = {
    directives.foreach { d =>
      d.directiveName match {
        // if this starts to grow extract directives and it's action into a separate class
        case "status" => releaseActorService.updateTaskStatusLine(d.taskId, d.payload)
        case "comment" => createComment(d.taskId, d.payload)
        case directiveName => logger.warn(s"Received unknown directive [$directiveName]")
      }
    }
  }
}


private[impl] trait InputContextFactory {
  def create(task: ContainerTask): JobInputContext = {
    val release = task.getRelease
    val releaseContext = ReleaseContext(
      release.getId,
      AutomatedTaskAsUserContext(
        release.getScriptUsername,
        release.getScriptUserPassword
      )
    )
    import scala.jdk.CollectionConverters._
    val taskDescriptor = task.getType.getDescriptor
    val pds = taskDescriptor.getPropertyDescriptors.asScala.filter(pd => Set(Task.CATEGORY_INPUT, Task.CATEGORY_OUTPUT, "script").contains(pd.getCategory()))
    val taskProperties = pds.map(pd => PropertyDefinition(pd.getName, propertyValue(task, pd), pd.getKind.toString, pd.getCategory, pd.isPassword))
    val taskContext = TaskContext(
      task.getId,
      task.getTaskType.toString,
      taskProperties.toList
    )
    JobInputContext(releaseContext, taskContext)
  }

  //noinspection ScalaStyle
  private def propertyValue(ci: ConfigurationItem, pd: PropertyDescriptor): AnyRef = {
    import java.util.{List => JList, Map => JMap, Set => JSet}
    import scala.jdk.CollectionConverters._
    val rawValue = pd.get(ci)
    if (rawValue != null) {
      pd.getKind match {
        case PropertyKind.BOOLEAN => rawValue
        case PropertyKind.INTEGER => rawValue
        case PropertyKind.STRING => rawValue
        case PropertyKind.ENUM => rawValue.toString
        case PropertyKind.DATE => rawValue // TODO check CiJson2Writer and what not (timezone etc)
        case PropertyKind.CI => ciValue(ci, pd)
        case PropertyKind.SET_OF_CI =>
          val ciCollection = rawValue.asInstanceOf[JSet[ConfigurationItem]]
          ciCollection.asScala.map(c => CiReference(c.getId, c.getType.toString))
        case PropertyKind.LIST_OF_CI =>
          val ciCollection = rawValue.asInstanceOf[JList[ConfigurationItem]]
          ciCollection.asScala.map(c => CiReference(c.getId, c.getType.toString))
        case PropertyKind.SET_OF_STRING => rawValue.asInstanceOf[JSet[String]].asScala.toSet
        case PropertyKind.LIST_OF_STRING => rawValue.asInstanceOf[JList[String]].asScala.toList
        case PropertyKind.MAP_STRING_STRING => rawValue.asInstanceOf[JMap[String, String]].asScala.toMap
      }
    } else {
      null
    }
  }

  private def ciValue(ci: ConfigurationItem, pd: PropertyDescriptor): AnyRef = {
    import scala.jdk.CollectionConverters._
    val rawValue = pd.get(ci).asInstanceOf[ConfigurationItem]
    val isConfiguration = pd.getReferencedType.isSubTypeOf(Type.valueOf(classOf[Configuration]))
    if (isConfiguration) {
      val ciDescriptor = rawValue.getType.getDescriptor
      val pds = ciDescriptor.getPropertyDescriptors.asScala
      val ciProperties = pds.map(pd => PropertyDefinition(pd.getName, propertyValue(rawValue, pd), pd.getKind.toString, pd.getCategory, pd.isPassword))
      ConfigurationContext(
        rawValue.getId,
        rawValue.getType.toString,
        ciProperties.toList
      )
    } else {
      CiReference(rawValue.getId, rawValue.getType.toString) // is it embedded, nested or what, is it a CI reference, like a container CI?
    }

  }
}

private[impl] trait InputContextSerialization extends Logging {

  def objectMapper: ObjectMapper

  def configurationService: ConfigurationService

  def serialize(inputContext: JobInputContext, runnerId: RunnerId): ContextData = {
    val inputContextJson = objectMapper.writeValueAsString(inputContext)
    encryptContextData(runnerId, inputContextJson)
  }

  private def encryptContextData(runnerId: RunnerId, inputContextJson: String): ContextData = {
    val config = configurationService.read(runnerId)
    config match {
      case runner: RemoteJobRunner =>
        ContextDataHelper.encryptData(runner, inputContextJson)
      case _ => PlainContextData(inputContextJson)
    }
  }
}

private[impl] trait OutputContextDeserialization extends Logging {

  def objectMapper: ObjectMapper

  def configurationService: ConfigurationService

  def deserialize(runnerId: RunnerId, output: ContextData): JobOutputContext = {
    output match {
      case plainData: PlainContextData =>
        objectMapper.readValue(plainData.data, classOf[JobOutputContext])
      case encryptedData: EncryptedContextData =>
        val config = configurationService.read(runnerId)
        config match {
          case runner: RemoteJobRunner =>
            val plainContextData = ContextDataHelper.decryptData(runner, encryptedData)
            objectMapper.readValue(plainContextData.data, classOf[JobOutputContext])
          case _ => throw new RuntimeException(s"Encrypted job data is only supported for remote job runners")
        }
    }
  }
}

object RunnerJobServiceImpl {
  case class CiReference(id: String, `type`: String) extends SerializableMsg

  case class PropertyDefinition(name: String, value: AnyRef, kind: String, category: String, password: Boolean) extends SerializableMsg {
    override def toString: String = {
      val valueAsString = if (password) "****" else value
      s"PropertyDefinition(name = $name, value = $valueAsString, kind = $kind, category = $category, password = $password)"
    }
  }

  case class ConfigurationContext(id: String, `type`: String, properties: List[PropertyDefinition]) extends SerializableMsg

  case class TaskContext(id: String, `type`: String, properties: List[PropertyDefinition]) extends SerializableMsg

  case class AutomatedTaskAsUserContext(username: String, password: String) extends SerializableMsg {
    override def toString: String = {
      s"AutomatedTaskAsUserContext(username = $username)"
    }
  }

  case class ReleaseContext(id: String, automatedTaskAsUser: AutomatedTaskAsUserContext) extends SerializableMsg

  case class JobInputContext(release: ReleaseContext, task: TaskContext) extends SerializableMsg

  // TODO add (de)serialization test for output context (if reportingRecords is not present at all, if it's null, if it's empty)
  case class JobOutputContext(exitCode: Int,
                              @(JsonDeserialize@field)(contentUsing = classOf[UntypedObjectDeserializer])
                              outputProperties: util.Map[String, AnyRef] = new util.HashMap[String, AnyRef](),
                              @(JsonDeserialize@field)(contentUsing = classOf[CiDeserializer])
                              reportingRecords: util.List[_ <: TaskReportingRecord] = new util.ArrayList[TaskReportingRecord]())
}
