package com.xebialabs.xlrelease.runner.impl

import com.codahale.metrics.annotation.Timed
import com.google.common.cache.{CacheBuilder, CacheLoader, LoadingCache}
import com.xebialabs.xlrelease.events.{EventListener, Subscribe}
import com.xebialabs.xlrelease.runner.domain.{RunnerId, RunnerToken}
import com.xebialabs.xlrelease.runner.repository.RunnerTokenRepository
import com.xebialabs.xlrelease.runner.service.RunnerTokenService
import com.xebialabs.xlrelease.service.BroadcastService
import com.xebialabs.xlrelease.utils.TokenGenerator
import org.springframework.stereotype.Service

import java.util.concurrent.TimeUnit

@Service
@EventListener
class RunnerTokenServiceImpl(broadcastService: BroadcastService,
                             runnerTokenRepository: RunnerTokenRepository)
  extends RunnerTokenService {

  private val runnerTokenCache: LoadingCache[String, Option[RunnerId]] = CacheBuilder.newBuilder
    .maximumSize(1000)
    .expireAfterWrite(30, TimeUnit.MINUTES)
    .build[String, Option[RunnerId]](new CacheLoader[String, Option[RunnerId]]() {
      override def load(tokenHash: String): Option[RunnerId] = {
        runnerTokenRepository.findByToken(tokenHash)
      }
    })

  @Timed
  override def createOrUpdateRunnerToken(runnerId: RunnerId): RunnerToken = {
    val token: String = TokenGenerator.generate()
    val tokenHash: String = TokenGenerator.hash(token)

    if(runnerTokenRepository.exists(runnerId)) {
      runnerTokenRepository.update(runnerId, tokenHash)
      broadcastService.broadcast(RunnerTokenUpdatedEvent(runnerId), publishEventOnSelf = true)
    } else {
      runnerTokenRepository.create(runnerId, tokenHash)
      broadcastService.broadcast(RunnerTokenCreatedEvent(runnerId), publishEventOnSelf = true)
    }
    RunnerToken(runnerId, token)
  }

  @Timed
  override def deleteRunnerToken(runnerId: RunnerId): Unit = {
    //TODO: delete runner tokens when the runner is deleted
    runnerTokenRepository.delete(runnerId)
    broadcastService.broadcast(RunnerTokenDeletedEvent(runnerId), publishEventOnSelf = true)
  }

  @Timed
  override def findByToken(tokenHash: String): Option[RunnerId] = {
    runnerTokenCache.get(tokenHash)
  }

  @Subscribe
  def onRunnerTokenEvent(runnerTokenEvent: RunnerTokenEvent): Unit = {
    runnerTokenCache.invalidateAll()
  }
}
