package com.xebialabs.xlrelease.runner.impl

import akka.actor.{Actor, ActorLogging}
import com.xebialabs.xlrelease.runner.domain._
import com.xebialabs.xlrelease.runner.impl.RunnerControlChannelActor.{CloseControlChannel, OpenControlChannel, SendCommand}
import com.xebialabs.xlrelease.runner.impl.RunnerProxyActor.{AddChannel, RemoveChannel}
import com.xebialabs.xlrelease.support.akka.SerializableMsg
import com.xebialabs.xlrelease.support.akka.spring.SpringActor

import javax.ws.rs.core.MediaType
import javax.ws.rs.sse.{OutboundSseEvent, Sse, SseEventSink}

@SpringActor
class RunnerControlChannelActor(runnerProxyFactory: RunnerProxyFactory) extends Actor with ActorLogging {

  override def receive: Receive = onMessage(None, None)

  private def onMessage(sink: Option[SseEventSink], sse: Option[Sse]): Receive = {
    case OpenControlChannel(runnerId, newSink, newSse) =>
      // no need to persist
      sink.foreach(_.close())
      context.become(onMessage(Option(newSink), Option(newSse)))
      // subscribe this channel as command channel to RunnerProxyActor
      runnerProxyFactory.create(()) ! AddChannel(runnerId, controlChannel = self)
    case CloseControlChannel(runnerId) =>
      // no need to persist
      sink.foreach(_.close())
      context.become(onMessage(None, None))
      // unsubscribe channel
      runnerProxyFactory.create(()) ! RemoveChannel(runnerId, controlChannel = self)
    case SendCommand(runnerId, command: PersistedCommand) =>
      // TODO.. sometimes sinks can be closed
      sink.foreach(_.send(createSseEvent(command, sse.map(_.newEventBuilder()).get)))
  }

  private def createSseEvent(persistedCommand: PersistedCommand, eventBuilder: OutboundSseEvent.Builder): OutboundSseEvent = {
    val command = persistedCommand.command
    val eventName = command.getClass.getSimpleName
    eventBuilder.name(eventName)
      .mediaType(MediaType.APPLICATION_JSON_TYPE)
      .id(persistedCommand.commandId)
      .data(persistedCommand)
      .build()
  }
}

object RunnerControlChannelActor {
  def actorName(runnerId: RunnerId): String = {
    s"control-channel-${runnerId.shortId()}"
  }

  sealed trait ChannelCommand extends SerializableMsg {
    def runnerId: RunnerId
  }

  case class OpenControlChannel(runnerId: RunnerId, newSink: SseEventSink, newSse: Sse) extends ChannelCommand

  case class CloseControlChannel(runnerId: RunnerId) extends ChannelCommand

  case class SendCommand(runnerId: RunnerId, persistedCommand: PersistedCommand) extends ChannelCommand
}
