package com.xebialabs.xlrelease.runner.impl

import akka.actor.{ActorRef, PoisonPill}
import akka.persistence.RecoveryCompleted
import com.xebialabs.xlrelease.actors.XlrPersistentActor
import com.xebialabs.xlrelease.runner.domain.{RunnerControlCommand, PersistedCommand, RunnerId}
import com.xebialabs.xlrelease.runner.impl.RunnerControlChannelActor.SendCommand
import com.xebialabs.xlrelease.runner.impl.RunnerProxyActor._
import com.xebialabs.xlrelease.support.akka.spring.SpringActor
import com.xebialabs.xlrelease.runner.domain._

import java.util.UUID
import scala.collection.mutable

@SpringActor
class RunnerProxyActor(runnerId: RunnerId) extends XlrPersistentActor {
  private var state: RunnerProxyState = Initial(runnerId, Map())

  private val openChannels: mutable.Set[ActorRef] = mutable.Set()

  override def persistenceId: String = self.path.name

  override def receiveRecover: Receive = {
    case evt: RunnerProxyEvent =>
      state = state.applyEvent(evt)
    case RecoveryCompleted =>
      log.debug(s"Recovery completed for $persistenceId. State is $state")
    // TODO ship all commands that are in the current state
    case msg =>
      log.warning(s"Can't recover message [$msg]")
  }

  override def receiveCommand: Receive = {
    case command: RunnerProxyCommand => handleCommand(command)
    case PoisonPill => context.stop(self)
    case unknown => log.error(s"Received unknown command $unknown")
  }

  private def handleCommand(command: RunnerProxyActor.RunnerProxyCommand): Unit = command match {
    case RunnerProxyActor.NewCommand(runnerId, command) =>
      val commandId = UUID.randomUUID().toString
      persist(CommandAdded(PersistedCommand(commandId, command)))(handleEvent)
    case RunnerProxyActor.ConfirmCommand(runnerId, commandId) =>
      persist(CommandConfirmed(commandId))(handleEvent)
    case RunnerProxyActor.AddChannel(runnerId, controlChannel) =>
      openChannels.add(controlChannel)
      // publish all commands in the state
      state.commands.foreach {
        case (commandId, command) => controlChannel ! SendCommand(runnerId, command)
      }

    case RunnerProxyActor.RemoveChannel(runnerId, controlChannel) =>
      openChannels.remove(controlChannel)
  }

  private def handleEvent(event: RunnerProxyEvent): Unit = {
    state = state.applyEvent(event)
    event match {
      case CommandAdded(command) =>
        openChannels.foreach(_ ! SendCommand(runnerId, command))
      case CommandConfirmed(_) =>
        // do nothing as command will be removed from internal state in applyEvent
        ()
    }
  }

}

object RunnerProxyActor {

  final val SHARDING_TYPE_NAME = "runner-proxy"

  def actorName(runnerId: RunnerId): String = {
    s"runner-proxy-${runnerId.shortId()}"
  }

  sealed trait RunnerProxyState {
    def applyEvent(event: RunnerProxyEvent): RunnerProxyState

    def commands: Map[String, PersistedCommand]
  }

  case class Initial(runnerId: RunnerId, commands: Map[String, PersistedCommand]) extends RunnerProxyState {
    override def applyEvent(event: RunnerProxyEvent): RunnerProxyState = event match {
      case CommandAdded(persistedCommand) =>
        copy(commands = commands + (persistedCommand.commandId -> persistedCommand))
      case CommandConfirmed(commandId) =>
        copy(commands = commands - commandId)
    }
  }

  // commands
  sealed trait RunnerProxyCommand {
    def runnerId: RunnerId
  }

  case class NewCommand(runnerId: RunnerId, command: RunnerControlCommand) extends RunnerProxyCommand

  case class ConfirmCommand(runnerId: RunnerId, commandId: String) extends RunnerProxyCommand

  case class AddChannel(runnerId: RunnerId, controlChannel: ActorRef) extends RunnerProxyCommand

  case class RemoveChannel(runnerId: RunnerId, controlChannel: ActorRef) extends RunnerProxyCommand

  // events
  sealed trait RunnerProxyEvent

  case class CommandAdded(persistedCommand: PersistedCommand) extends RunnerProxyEvent

  case class CommandConfirmed(commandId: String) extends RunnerProxyEvent

}
