package com.xebialabs.xlrelease.runner.impl

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.domain.Configuration
import com.xebialabs.xlrelease.domain.runner.RemoteJobRunner
import com.xebialabs.xlrelease.repository.Ids.{CUSTOM_CONFIGURATION_ROOT, SEPARATOR}
import com.xebialabs.xlrelease.runner.api.JobRunnerResource
import com.xebialabs.xlrelease.runner.api.JobRunnerResource.{RegisterRequest, RegisterResponse}
import com.xebialabs.xlrelease.runner.domain._
import com.xebialabs.xlrelease.runner.service.RunnerJobService
import com.xebialabs.xlrelease.scheduler.storage.spring.StorageConfiguration.URI_SCHEME_LOCAL_STORAGE
import com.xebialabs.xlrelease.security.PermissionChecker
import com.xebialabs.xlrelease.service.ConfigurationService
import com.xebialabs.xlrelease.storage.domain.LogEntry
import com.xebialabs.xlrelease.storage.service.StorageService
import org.springframework.stereotype.Controller

import javax.ws.rs.sse.{Sse, SseEventSink}
import scala.jdk.CollectionConverters._

@Controller
class JobRunnerResourceImpl(permissionChecker: PermissionChecker,
                            runnerJobService: RunnerJobService,
                            runnerControlService: RunnerControlService,
                            storageService: StorageService,
                            configurationService: ConfigurationService,
                           ) extends JobRunnerResource {

  // TODO any of these messages can fail or timeout - handle errors on the client side
  //  i.e. see how we can get and log meaningful data
  //  remember we cannot use debugger and resteasy hides interesting details (like when deserialization fails)

  override def getJob(runnerId: String): JobData = {
    val jobData = runnerJobService.getJob(runnerId)
    jobData
  }

  override def failJob(jobId: JobId): Unit = {
    runnerJobService.failJob(jobId)
  }

  override def finishJob(jobResult: JobResult): Unit = {
    runnerJobService.finishJob(jobResult)
  }

  override def log(logEntry: LogEntry): Unit = {
    // TODO temporary storing entry here, but maybe we should do it inside of the service
    val workerLogEntry = logEntry.copy(uriScheme = URI_SCHEME_LOCAL_STORAGE)
    val storedEntryUri = storageService.store(workerLogEntry)
    runnerJobService.log(logEntry)
  }

  override def executeDirectives(directives: Seq[JobDirective]): Unit = {
    runnerJobService.executeDirectives(directives)
  }

  override def getRegistrationToken(): String = {
    // TODO implement registration token properly
    //  - needs to be secured, has to expire registration token etc
    // short lived registration token used to create longer lived PAT
    // if user has permission to register a runner - let it create a token that will allow it to register runner
    //    permissionChecker.checkIsAllowedToCreateRegistrationToken()
    "registration-token"
  }

  override def register(registerRequest: RegisterRequest): RegisterResponse = {
    // TODO create runner - we should be authenticated using registration token at this point
    //  authenticated with registration token and uses registration token to generate PAT
    permissionChecker.checkIsAllowedToRegisterRunner()
    val itemId = runnerId(registerRequest.uuid)
    val runner = if (configurationService.exists(itemId)) {
      configurationService.read(itemId).asInstanceOf[RemoteJobRunner]
    } else {
      val newRunner = Type.valueOf(classOf[RemoteJobRunner]).getDescriptor.newInstance(itemId).asInstanceOf[RemoteJobRunner]
      newRunner.setTitle(s"runner - ${registerRequest.uuid}")
      newRunner.setCapabilityTags(registerRequest.capabilities.asJava)
      configurationService.createOrUpdate(newRunner)
      newRunner
    }
    RegisterResponse(runner.getId(), runner.isEnabled(), runner.capacity)
  }

  private def runnerId(uuid: String): String = {
    val parentId = CUSTOM_CONFIGURATION_ROOT
    val typePrefix = Type.valueOf(classOf[Configuration]).getName
    val uuidString = uuid.filterNot(_ == '-')
    s"$parentId$SEPARATOR$typePrefix$uuidString"
  }

  override def commands(runnerId: RunnerId, sink: SseEventSink, sse: Sse): Unit = {
    runnerControlService.watch(runnerId, sink, sse)
  }

  override def confirmCommand(runnerId: String, commandId: String): Unit = {
    runnerControlService.confirmCommand(runnerId, commandId)
  }
}
