package com.xebialabs.xlrelease.risk

import com.xebialabs.xlrelease.actors.StopProtocol.{RequestStop, Stopped}
import com.xebialabs.xlrelease.actors.{ManagedActor, releaseId2ActorName}
import com.xebialabs.xlrelease.domain.Release
import com.xebialabs.xlrelease.risk.service.RiskService
import com.xebialabs.xlrelease.support.pekko.spring.SpringActor
import org.apache.pekko.actor.{Actor, ActorRef, PoisonPill, Terminated}
import org.apache.pekko.event.slf4j.SLF4JLogging


@SpringActor
class ReleaseRiskSupervisorActor(riskService: RiskService) extends Actor with SLF4JLogging {

  import ReleaseRiskSupervisorActor._

  private var watchedActorsCount: Int = 0

  override def postStop(): Unit = {
    log.debug(s"Stopping ${self.path}")
    super.postStop()
  }

  override def receive: Receive = handleMsgs(Set[String]())

  private def handleMsgs(watchedReleases: Set[String]): Receive = {
    case Watch(releaseId) =>
      log.debug("Release with id '{}' is watched for risk", releaseId)

      context.become(handleMsgs(watchedReleases + releaseId2ActorName(releaseId)))
    case UnWatch(releaseId) =>
      log.debug("Release with id '{}' is not watched anymore for risk", releaseId)
      context.child(releaseId2ActorName(releaseId)).foreach(_ ! PoisonPill)
      context.become(handleMsgs(watchedReleases - releaseId))
    case msg@CalculateReleaseRisk(_, release) =>
      if (watchedReleases.contains(releaseId2ActorName(release.getId))) {
        createOrFind(release.getId) forward (msg)
      } else {
        log.warn("Release with id '{}' is currently not watched for risk", release.getId)
      }
    case RequestStop(correlationId) =>
      context.children.foreach(_ ! PoisonPill)
      if (watchedActorsCount == 0) {
        sender() ! Stopped(correlationId)
        context.become(receive)
      } else {
        context.become(stopping(sender(), correlationId))
      }
    case Terminated(_) =>
      if (watchedActorsCount > 0) {
        watchedActorsCount -= 1
      }
  }

  private def stopping(monitoringActor: ActorRef, correlationId: String): Receive = {
    case Terminated(_) =>
      if (watchedActorsCount > 0) {
        watchedActorsCount -= 1
      }
      if (watchedActorsCount == 0) {
        monitoringActor ! Stopped(correlationId)
        context.become(receive)
      }
    case Watch(releaseId) =>
      log.warn("Risk supervisor is terminating release with id '{}' cannot be watched", releaseId)
    case UnWatch(releaseId) =>
      log.warn("Risk supervisor is terminating release with id '{}' cannot be un-watched", releaseId)
    case CalculateReleaseRisk(_, release) =>
      log.warn("Risk supervisor is terminating risk cannot be calculated for release with id '{}'", release.getId)
    case msg =>
      log.warn("Risk supervisor is terminating. Message '{}' cannot be processed ", msg)
  }

  private def createOrFind(releaseId: String): ActorRef = {
    context.child(releaseId2ActorName(releaseId)).getOrElse({
      val riskActor = context.actorOf(ReleaseRiskActor.props(releaseId, riskService), releaseId2ActorName(releaseId))
      watchedActorsCount += 1
      context.watch(riskActor)
    })
  }
}

object ReleaseRiskSupervisorActor {
  type ReleaseRiskSupervisorManagedActor = ManagedActor[ReleaseRiskSupervisorActor]

  // this needs to be serializable? not really - because it runs on every node and is not distributed! :D
  case class CalculateReleaseRisk(eventType: Class[_], release: Release)

  case class Watch(releaseId: String)

  case class UnWatch(releaseId: String)
}