package com.xebialabs.xlrelease.risk.repository

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.domain.ReleaseExtension
import com.xebialabs.xlrelease.repository.Ids.{getName, getParentId, releaseIdFrom}
import com.xebialabs.xlrelease.repository.ReleaseExtensionsRepository.ReleaseExtensionSpecializedRepository
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.CiId
import com.xebialabs.xlrelease.repository.sql.persistence.{CiUid, ReleasePersistence}
import com.xebialabs.xlrelease.risk.builder.RiskBuilder
import com.xebialabs.xlrelease.risk.domain.{Risk, RiskAssessment}
import com.xebialabs.xlrelease.risk.repository.sql.persistence.RiskPersistence
import com.xebialabs.xlrelease.service.ReleaseService
import com.xebialabs.xlrelease.utils.Diff
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Repository

import scala.jdk.CollectionConverters._


@Repository
@IsTransactional
class RiskRepository @Autowired()(riskPersistence: RiskPersistence,
                                  releasePersistence: ReleasePersistence,
                                  releaseService: ReleaseService
                                 )
  extends ReleaseExtensionSpecializedRepository[Risk]
    with Logging {

  override val name: String = Risk.RISK_PREFIX

  override def exists(riskId: String): Boolean = {
    logger.debug(s"exists($riskId)")
    releasePersistence.findUidByReleaseId(releaseIdFrom(riskId))
      .exists(riskPersistence.exists)
  }

  override def read(riskId: String): Option[Risk] = {
    logger.debug(s"read($riskId)")
    withRelease0(releaseIdFrom(riskId)) { releaseUid =>
      riskPersistence.read(releaseUid).map { risk =>
        risk.setId(riskId)
        val riskAssessments = riskPersistence.readAllAssessments(releaseUid)
        riskAssessments.foreach(_.setRisk(risk))
        risk.setRiskAssessments(riskAssessments.asJava)
        risk
      }
    }.flatten
  }


  override def create(risk: Risk): Risk = {
    logger.debug(s"create(${risk.getId})")
    withRelease(releaseIdFrom(risk.getId)) { releaseUid =>
      riskPersistence.create(releaseUid, risk.getScore, risk.getTotalScore)
      riskPersistence.createOrUpdateAssessments(releaseUid, risk.getRiskAssessments.asScala.toSeq)
      risk
    }
  }

  override def update(risk: Risk): Boolean = {
    logger.debug(s"update(${risk.getId})")
    withRelease(releaseIdFrom(risk.getId)) { releaseUid =>
      updateRiskAssessments(releaseUid, risk.getRiskAssessments.asScala.toSeq)
      riskPersistence.update(releaseUid, risk.getScore, risk.getTotalScore) == 1
    }
  }

  override def delete(riskId: String): Boolean = {
    logger.debug(s"delete($riskId)")
    withRelease(releaseIdFrom(riskId)) { releaseUid =>
      val numDeleted = riskPersistence.deleteAllAssessments(releaseUid)
      val deleted = riskPersistence.delete(releaseUid)
      numDeleted > 0 && deleted == 1
    }
  }

  def findByIdOrDefault(riskId: String): Risk = {
    logger.debug(s"findByIdOrDefault($riskId)")
    read(riskId).orElse {
      val releaseId = releaseIdFrom(riskId)
      if (releaseService.isArchived(releaseId)) {
        getExtensionFromArchivedRelease(releaseId)
      } else {
        None
      }
    }.getOrElse(defaultRisk(riskId))
  }

  def findRiskAssessmentById(riskAssessmentId: String): RiskAssessment =  {
    logger.debug(s"findRiskAssessmentById($riskAssessmentId)")
    val risk = findByIdOrDefault(getParentId(riskAssessmentId))
    risk.getRiskAssessments.asScala.find(ra => ra.getId == riskAssessmentId).getOrElse {
      throw new NotFoundException(s"RiskAssessment $riskAssessmentId not found.")
    }
  }

  def defaultRisk(riskId: String): Risk =
    RiskBuilder.newRisk()
      .withId(riskId)
      .withScore(0)
      .withTotalScore(0)
      .withAssessments()
      .build

  protected def updateRiskAssessments(releaseUid: CiUid, riskAssessments: Seq[RiskAssessment]): Boolean = {
    logger.debug(s"updateRiskAssessments($releaseUid, ${riskAssessments.map(_.getId)})")
    val original = riskPersistence.readAllAssessments(releaseUid)
    val diff = Diff(
      original.map(ra => ra.getRiskAssessorId -> ra).toMap,
      riskAssessments.map(ra => ra.getRiskAssessorId -> ra).toMap
    )
    val numDeleted = riskPersistence.deleteAssessments(releaseUid, diff.deletedKeys.toSeq)
    val numUpdated = riskPersistence.updateAssessments(releaseUid, diff.updatedValues.toSeq)
    val numCreated = riskPersistence.createOrUpdateAssessments(releaseUid, diff.newValues.toSeq)
    val num = numCreated + numUpdated - numDeleted
    riskAssessments.size == num
  }

  protected def getExtensionFromArchivedRelease(releaseId: CiId): Option[Risk] = {
    logger.debug(s"getExtensionFromArchivedRelease($releaseId)")
    val release = releaseService.findByIdInArchive(releaseId)
    release.getProperty[java.util.List[ReleaseExtension]]("extensions").asScala
      .collectFirst {
        case risk: Risk if getName(risk.getId) == name => risk
      }
  }

  protected def withRelease0[A](ciId: CiId)(f: CiUid => A): Option[A] = {
    releasePersistence.findUidByReleaseId(ciId).map(f)
  }

  protected def withRelease[A](ciId: CiId)(f: CiUid => A): A = {
    withRelease0(ciId)(f).getOrElse {
      throw new NotFoundException(s"Release [$ciId] not found")
    }
  }

}
