package com.xebialabs.xlrelease.risk.spring.config;

import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.scheduling.annotation.EnableAsync;

import com.xebialabs.xlrelease.config.XlrConfig;
import com.xebialabs.xlrelease.scheduler.RestartableExecutorService;
import com.xebialabs.xlrelease.scheduler.XlrExecutors;
import com.xebialabs.xlrelease.service.FeatureService;

import io.micrometer.core.instrument.MeterRegistry;

@Configuration
@EnableAsync
public class RiskConfiguration implements FeatureService {
    private static final String RISK_CALCULATION = "risk-calculation";
    private static final String RISK_CALCULATION_BATCH = "risk-calculation-batch";
    public static final String RISK_CALCULATION_EXECUTOR = "riskCalculationExecutor";
    private final XlrConfig xlrConfig;

    private volatile boolean enabled = true;
    private final MeterRegistry meterRegistry;

    public RiskConfiguration(final MeterRegistry meterRegistry, final XlrConfig xlrConfig) {
        this.meterRegistry = meterRegistry;
        this.xlrConfig = xlrConfig;
    }

    @Bean(name = RISK_CALCULATION_EXECUTOR)
    public RestartableExecutorService riskCalculationExecutor() {
        var maxThreads = xlrConfig.executors().riskCalculation().maxThreadsCount();
        var shutdownTimeout = xlrConfig.executors().riskCalculation().shutdownTimeout();
        return XlrExecutors.newFixedThreadPool(RISK_CALCULATION, maxThreads, meterRegistry, shutdownTimeout);
    }

    @Bean
    public RestartableExecutorService riskCalculationBatchExecutor() {
        var maxThreads = xlrConfig.executors().riskCalculationBatch().maxThreadsCount();
        var shutdownTimeout = xlrConfig.executors().riskCalculationBatch().shutdownTimeout();
        return XlrExecutors.newFixedThreadPool(RISK_CALCULATION_BATCH, maxThreads, meterRegistry, shutdownTimeout);
    }

    @Override
    public String serviceName() {
        return "risk execution service";
    }

    @Override
    public void start() {
        if (!enabled) {
            riskCalculationExecutor().start();
            riskCalculationBatchExecutor().start();
        }
        enabled = true;
    }

    @Override
    public void stop() {
        if (enabled) {
            riskCalculationExecutor().stop();
            riskCalculationBatchExecutor().stop();
        }
        enabled = false;
    }

    @Override
    public boolean isRunning() {
        return enabled;
    }
}
