package com.xebialabs.xlrelease.risk

import com.xebialabs.xlrelease.actors.{ManagedActor, releaseId2ActorName}
import com.xebialabs.xlrelease.domain.Release
import com.xebialabs.xlrelease.risk.service.RiskService
import com.xebialabs.xlrelease.support.pekko.spring.SpringActor
import org.apache.pekko.actor.{Actor, ActorRef, PoisonPill}
import org.apache.pekko.event.slf4j.SLF4JLogging


@SpringActor
class ReleaseRiskSupervisorActor(riskService: RiskService) extends Actor with SLF4JLogging {

  import ReleaseRiskSupervisorActor._

  override def postStop(): Unit = {
    log.debug(s"Stopping ${self.path}")
    super.postStop()
  }

  override def receive: Receive = handleMsgs(Set[String]())

  private def handleMsgs(watchedReleases: Set[String]): Receive = {
    case Watch(releaseId) =>
      log.debug("Release with id '{}' is watched for risk", releaseId)
      context.become(handleMsgs(watchedReleases + releaseId2ActorName(releaseId)))
    case UnWatch(releaseId) =>
      log.debug("Release with id '{}' is not watched anymore for risk", releaseId)
      context.child(releaseId2ActorName(releaseId)).foreach(_ ! PoisonPill)
      context.become(handleMsgs(watchedReleases - releaseId))
    case msg@CalculateReleaseRisk(_, release) =>
      if (watchedReleases.contains(releaseId2ActorName(release.getId))) {
        createOrFind(release.getId) forward (msg)
      } else {
        log.warn("Release with id '{}' is currently not watched for risk", release.getId)
      }
  }

  private def createOrFind(releaseId: String): ActorRef = {
    context.child(releaseId2ActorName(releaseId)).getOrElse({
      context.actorOf(ReleaseRiskActor.props(releaseId, riskService), releaseId2ActorName(releaseId))
    })
  }
}

object ReleaseRiskSupervisorActor {
  type ReleaseRiskSupervisorManagedActor = ManagedActor[ReleaseRiskSupervisorActor]


  // this needs to be serializable? not really - because it runs on every node and is not distributed! :D
  case class CalculateReleaseRisk(eventType: Class[_], release: Release)

  case class Watch(releaseId: String)

  case class UnWatch(releaseId: String)
}