package com.xebialabs.xlrelease.risk.repository.sql.persistence

import com.xebialabs.xlplatform.repository.sql.DatabaseType
import com.xebialabs.xlrelease.db.sql.SqlBuilder
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.domain.id.CiUid
import com.xebialabs.xlrelease.repository.sql.persistence.CiId._
import com.xebialabs.xlrelease.repository.sql.persistence.PersistenceSupport
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.params
import com.xebialabs.xlrelease.risk.domain.{Risk, RiskAssessment}
import com.xebialabs.xlrelease.risk.repository.sql.persistence.Schema.{RISKS, RISK_ASSESSMENTS}
import com.xebialabs.xlrelease.spring.config.SqlConfiguration
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.dao.DuplicateKeyException
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}
import org.springframework.stereotype.Repository
import org.springframework.transaction.support.{DefaultTransactionDefinition, TransactionTemplate}
import org.springframework.transaction.{PlatformTransactionManager, TransactionDefinition, TransactionStatus}
import spray.json._

import java.sql.ResultSet
import scala.jdk.CollectionConverters._

@Repository
@IsTransactional
class RiskPersistence(@Qualifier("xlrRepositoryTransactionManager") txManager: PlatformTransactionManager)
                     (implicit
                      @Qualifier("xlrRepositoryJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                      @Qualifier("xlrRepositorySqlDialect") val dialect: SqlBuilder.Dialect)
  extends PersistenceSupport
    with DefaultJsonProtocol
    with Logging {

  private val STMT_EXISTS: String =
    s"""SELECT 1 FROM ${RISKS.TABLE} WHERE ${RISKS.RELEASE_UID} = :releaseUid"""

  def exists(releaseUid: CiUid): Boolean =
    findOne(sqlQuery(STMT_EXISTS, params("releaseUid" -> releaseUid), _.getInt(1) == 1)).getOrElse(false)

  def createOrUpdate(releaseUid: CiUid, score: Int, totalScore: Int): Int = {
    logger.debug(s"createOrUpdate($releaseUid, $score, $totalScore)")
    try {
      withNewTransation("createRisk") { _ =>
        create(releaseUid, score, totalScore)
      }
    } catch {
      case e: DuplicateKeyException =>
        withNewTransation("updateRisk") { _ =>
          update(releaseUid, score, totalScore)
        }
    }
  }

  def createOrUpdateAssessments(releaseUid: CiUid, riskAssessments: Seq[RiskAssessment]): Int = {
    logger.debug(s"createOrUpdateAssessments($releaseUid, ${riskAssessments.map(_.getId)})")
    sqlBatchWithContent(createOrUpdateAssessment,
      riskAssessments.map(ra =>
        params(
          "releaseUid" -> releaseUid,
          "assessorId" -> ra.getRiskAssessorId,
          "score" -> ra.getScore,
          "headline" -> ra.getHeadline,
          "icon" -> ra.getIcon
        ) -> ("messages" -> ra.getMessages.asScala.toList.toJson.compactPrint)
      )
    ).sum
  }

  private val STMT_READ: String =
    s"""|SELECT
        |   ${RISKS.RELEASE_UID},
        |   ${RISKS.SCORE},
        |   ${RISKS.TOTAL_SCORE}
        |FROM ${RISKS.TABLE}
        |WHERE ${RISKS.RELEASE_UID} = :releaseUid""".stripMargin

  def read(releaseUid: CiUid): Option[Risk] = {
    logger.debug(s"read($releaseUid)")
    findOne(
      sqlQuery(STMT_READ, params("releaseUid" -> releaseUid), riskMapper)
    )
  }

  private val STMT_READ_ASSESSMENT: String =
    s"""|SELECT
        |   ${RISK_ASSESSMENTS.ASSESSOR_ID},
        |   ${RISK_ASSESSMENTS.SCORE},
        |   ${RISK_ASSESSMENTS.HEADLINE},
        |   ${RISK_ASSESSMENTS.MESSAGES},
        |   ${RISK_ASSESSMENTS.ICON}
        |FROM ${RISK_ASSESSMENTS.TABLE}
        |WHERE ${RISK_ASSESSMENTS.RELEASE_UID} = :releaseUid
     """.stripMargin

  def readAllAssessments(releaseUid: CiUid): Seq[RiskAssessment] = {
    logger.debug(s"readAssessments($releaseUid)")
    sqlQuery(STMT_READ_ASSESSMENT, params("releaseUid" -> releaseUid), riskAssessmentMapper).toSeq
  }

  private val STMT_UPDATE: String =
    s"""|UPDATE ${RISKS.TABLE}
        |SET
        |   ${RISKS.SCORE} = :score,
        |   ${RISKS.TOTAL_SCORE} = :totalScore
        |WHERE ${RISKS.RELEASE_UID} = :releaseUid""".stripMargin

  def update(releaseUid: CiUid, score: Int, totalScore: Int): Int = {
    logger.debug(s"update($releaseUid, $score, $totalScore)")
    sqlUpdate(STMT_UPDATE,
      params(
        "releaseUid" -> releaseUid,
        "score" -> score,
        "totalScore" -> totalScore
      ),
      identity
    )
  }

  private val STMT_UPDATE_ASSESSMENT: String =
    s"""|UPDATE ${RISK_ASSESSMENTS.TABLE}
        |SET
        |   ${RISK_ASSESSMENTS.SCORE} = :score,
        |   ${RISK_ASSESSMENTS.HEADLINE} = :headline,
        |   ${RISK_ASSESSMENTS.MESSAGES} = :messages,
        |   ${RISK_ASSESSMENTS.ICON} = :icon
        |WHERE ${RISK_ASSESSMENTS.RELEASE_UID} = :releaseUid AND
        |      ${RISK_ASSESSMENTS.ASSESSOR_ID} = :assessorId""".stripMargin

  def updateAssessments(releaseUid: CiUid, riskAssessments: Seq[RiskAssessment]): Int = {
    logger.debug(s"updateAssessments($releaseUid, ${riskAssessments.map(_.getRiskAssessorId)})")
    sqlBatchWithContent(STMT_UPDATE_ASSESSMENT,
      riskAssessments.map(ra =>
        params(
          "releaseUid" -> releaseUid,
          "assessorId" -> ra.getRiskAssessorId,
          "score" -> ra.getScore,
          "headline" -> ra.getHeadline,
          "icon" -> ra.getIcon
        ) -> ("messages" -> ra.getMessages.asScala.toList.toJson.compactPrint)
      )
    ).sum
  }

  private val STMT_DELETE: String =
    s"""DELETE FROM ${RISKS.TABLE} WHERE ${RISKS.RELEASE_UID} = :releaseUid"""

  def delete(releaseUid: CiUid): Int = {
    logger.debug(s"delete($releaseUid)")
    sqlUpdate(STMT_DELETE, params("releaseUid" -> releaseUid), identity)
  }

  private val STMT_DELETE_ASSESSMENTS: String =
    s"""DELETE FROM ${RISK_ASSESSMENTS.TABLE} WHERE ${RISK_ASSESSMENTS.RELEASE_UID} = :releaseUid"""

  def deleteAllAssessments(releaseUid: CiUid): Int = {
    logger.debug(s"deleteAllAssessments($releaseUid)")
    sqlUpdate(STMT_DELETE_ASSESSMENTS, params("releaseUid" -> releaseUid), identity)
  }

  private val STMT_DELETE_ASSESSMENT: String =
    STMT_DELETE_ASSESSMENTS + s" AND ${RISK_ASSESSMENTS.ASSESSOR_ID} = :assessorId"

  def deleteAssessments(releaseUid: CiUid, riskAssessorIds: Seq[CiId]): Int =
    sqlBatch(STMT_DELETE_ASSESSMENT,
      riskAssessorIds.map(riskAssessorId =>
        params(
          "releaseUid" -> releaseUid,
          "assessorId" -> riskAssessorId
        )
      ).toSet
    ).sum

  private val riskMapper: RowMapper[Risk] = (rs: ResultSet, _: Int) => {
    val risk = new Risk
    risk.setScore(rs.getInt(RISKS.SCORE))
    risk.setTotalScore(rs.getInt(RISKS.TOTAL_SCORE))
    risk
  }

  private val riskAssessmentMapper: RowMapper[RiskAssessment] = (rs: ResultSet, _: Int) => {
    val riskAssessment = new RiskAssessment
    riskAssessment.setRiskAssessorId(rs.getString(RISK_ASSESSMENTS.ASSESSOR_ID))
    riskAssessment.setScore(rs.getInt(RISK_ASSESSMENTS.SCORE))
    riskAssessment.setHeadline(rs.getString(RISK_ASSESSMENTS.HEADLINE))
    riskAssessment.setIcon(rs.getString(RISK_ASSESSMENTS.ICON))
    val messages = decompress(rs.getBinaryStream(RISK_ASSESSMENTS.MESSAGES))
      .parseJson
      .convertTo[List[String]]
    riskAssessment.setMessages(messages.asJava)
    riskAssessment
  }

  private def createOrUpdateAssessment: String = {
    SqlConfiguration.getDatabaseType(dialect) match {
      case DatabaseType.MySQL => STMT_CREATE_OR_UPDATE_ASSESSMENT_MYSQL
      case DatabaseType.Postgres => STMT_CREATE_OR_UPDATE_ASSESSMENT_POSTGRES
      case DatabaseType.H2 => STMT_CREATE_OR_UPDATE_ASSESSMENT_H2
      case DatabaseType.Oracle => STMT_CREATE_OR_UPDATE_ASSESSMENT_ORACLE
      case DatabaseType.SQLServer => STMT_CREATE_OR_UPDATE_ASSESSMENT_USING_MERGE + ";"
      case DatabaseType.DB2 => STMT_CREATE_OR_UPDATE_ASSESSMENT_USING_MERGE
      case DatabaseType.Derby => STMT_CREATE_OR_UPDATE_ASSESSMENT_DERBY
    }
  }

  private val STMT_CREATE: String =
    s"""|INSERT INTO ${RISKS.TABLE} (
        |   ${RISKS.RELEASE_UID},
        |   ${RISKS.SCORE},
        |   ${RISKS.TOTAL_SCORE}
        |) VALUES (
        |   :releaseUid,
        |   :score,
        |   :totalScore
        |)""".stripMargin

  def create(releaseUid: CiUid, score: Int, totalScore: Int): Int = {
    logger.debug(s"create($releaseUid, $score, $totalScore)")
    sqlUpdate(STMT_CREATE,
      params(
        "releaseUid" -> releaseUid,
        "score" -> score,
        "totalScore" -> totalScore
      ),
      identity
    )
  }

  private val STMT_CREATE_OR_UPDATE_ASSESSMENT_MYSQL: String =
    s"""|INSERT INTO ${RISK_ASSESSMENTS.TABLE} (
        |   ${RISK_ASSESSMENTS.RELEASE_UID},
        |   ${RISK_ASSESSMENTS.ASSESSOR_ID},
        |   ${RISK_ASSESSMENTS.SCORE},
        |   ${RISK_ASSESSMENTS.HEADLINE},
        |   ${RISK_ASSESSMENTS.MESSAGES},
        |   ${RISK_ASSESSMENTS.ICON}
        |) VALUES (
        |   :releaseUid,
        |   :assessorId,
        |   :score,
        |   :headline,
        |   :messages,
        |   :icon
        |) ON DUPLICATE KEY UPDATE
        |   ${RISK_ASSESSMENTS.SCORE} = :score,
        |   ${RISK_ASSESSMENTS.HEADLINE} = :headline,
        |   ${RISK_ASSESSMENTS.MESSAGES} = :messages,
        |   ${RISK_ASSESSMENTS.ICON} = :icon
        |""".stripMargin

  private val STMT_CREATE_OR_UPDATE_ASSESSMENT_POSTGRES: String =
    s"""|INSERT INTO ${RISK_ASSESSMENTS.TABLE} (
        |   ${RISK_ASSESSMENTS.RELEASE_UID},
        |   ${RISK_ASSESSMENTS.ASSESSOR_ID},
        |   ${RISK_ASSESSMENTS.SCORE},
        |   ${RISK_ASSESSMENTS.HEADLINE},
        |   ${RISK_ASSESSMENTS.MESSAGES},
        |   ${RISK_ASSESSMENTS.ICON}
        |) VALUES (
        |   :releaseUid,
        |   :assessorId,
        |   :score,
        |   :headline,
        |   :messages,
        |   :icon
        |) ON CONFLICT (${RISK_ASSESSMENTS.RELEASE_UID}, ${RISK_ASSESSMENTS.ASSESSOR_ID}) DO UPDATE SET
        |   ${RISK_ASSESSMENTS.SCORE} = :score,
        |   ${RISK_ASSESSMENTS.HEADLINE} = :headline,
        |   ${RISK_ASSESSMENTS.MESSAGES} = :messages,
        |   ${RISK_ASSESSMENTS.ICON} = :icon
        |""".stripMargin

  private val STMT_CREATE_OR_UPDATE_ASSESSMENT_H2 = {
    s"""
       |MERGE INTO ${RISK_ASSESSMENTS.TABLE} KEY(${RISK_ASSESSMENTS.RELEASE_UID}, ${RISK_ASSESSMENTS.ASSESSOR_ID})
       |VALUES (:releaseUid, :assessorId, :score, :headline, :messages, :icon)
       |""".stripMargin
  }

  private val STMT_CREATE_OR_UPDATE_ASSESSMENT_ORACLE: String = {
    s"""
       |MERGE INTO ${RISK_ASSESSMENTS.TABLE}
       |    USING DUAL
       |    ON (${RISK_ASSESSMENTS.RELEASE_UID} = :releaseUid AND ${RISK_ASSESSMENTS.ASSESSOR_ID} = :assessorId)
       |    WHEN MATCHED THEN UPDATE SET
       |      ${RISK_ASSESSMENTS.SCORE} = :score,
       |      ${RISK_ASSESSMENTS.HEADLINE} = :headline,
       |      ${RISK_ASSESSMENTS.MESSAGES} = :messages,
       |      ${RISK_ASSESSMENTS.ICON} = :icon
       |    WHEN NOT MATCHED THEN INSERT
       |      (${RISK_ASSESSMENTS.RELEASE_UID}, ${RISK_ASSESSMENTS.ASSESSOR_ID}, ${RISK_ASSESSMENTS.SCORE},
       |      ${RISK_ASSESSMENTS.HEADLINE}, ${RISK_ASSESSMENTS.MESSAGES}, ${RISK_ASSESSMENTS.ICON})
       |      VALUES (:releaseUid, :assessorId, :score, :headline, :messages, :icon)
       |""".stripMargin
  }

  private val STMT_CREATE_OR_UPDATE_ASSESSMENT_USING_MERGE: String = {
    s"""
       |MERGE INTO ${RISK_ASSESSMENTS.TABLE} AS T
       |    USING (VALUES(:releaseUid, :assessorId)) AS S (${RISK_ASSESSMENTS.RELEASE_UID},
       |    ${RISK_ASSESSMENTS.ASSESSOR_ID})
       |    ON (T.${RISK_ASSESSMENTS.RELEASE_UID} = S.${RISK_ASSESSMENTS.RELEASE_UID} AND
       |     T.${RISK_ASSESSMENTS.ASSESSOR_ID} = S.${RISK_ASSESSMENTS.ASSESSOR_ID})
       |    WHEN MATCHED THEN UPDATE SET
       |      ${RISK_ASSESSMENTS.SCORE} = :score,
       |      ${RISK_ASSESSMENTS.HEADLINE} = :headline,
       |      ${RISK_ASSESSMENTS.MESSAGES} = :messages,
       |      ${RISK_ASSESSMENTS.ICON} = :icon
       |    WHEN NOT MATCHED THEN INSERT
       |      (${RISK_ASSESSMENTS.RELEASE_UID}, ${RISK_ASSESSMENTS.ASSESSOR_ID}, ${RISK_ASSESSMENTS.SCORE},
       |      ${RISK_ASSESSMENTS.HEADLINE}, ${RISK_ASSESSMENTS.MESSAGES}, ${RISK_ASSESSMENTS.ICON})
       |      VALUES (:releaseUid, :assessorId, :score, :headline, :messages, :icon)
       |""".stripMargin
  }

  private val STMT_CREATE_OR_UPDATE_ASSESSMENT_DERBY: String = {
    s"""
       |MERGE INTO ${RISK_ASSESSMENTS.TABLE} AS T
       |    USING SYSIBM.SYSDUMMY1
       |    ON (T.${RISK_ASSESSMENTS.RELEASE_UID} = :releaseUid AND
       |      T.${RISK_ASSESSMENTS.ASSESSOR_ID} = :assessorId)
       |    WHEN MATCHED THEN UPDATE SET
       |      ${RISK_ASSESSMENTS.SCORE} = :score,
       |      ${RISK_ASSESSMENTS.HEADLINE} = :headline,
       |      ${RISK_ASSESSMENTS.MESSAGES} = :messages,
       |      ${RISK_ASSESSMENTS.ICON} = :icon
       |    WHEN NOT MATCHED THEN INSERT
       |      VALUES (:releaseUid, :assessorId, :score, :headline, :messages, :icon)
       |""".stripMargin
  }

  private def withNewTransation[T](txName: String)(block: TransactionStatus => T) = {
    val txDef = new DefaultTransactionDefinition
    txDef.setName(txName)
    txDef.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW)
    txDef.setIsolationLevel(TransactionDefinition.ISOLATION_READ_COMMITTED)
    val template = new TransactionTemplate(txManager, txDef)

    template.execute[T](ts => block(ts))
  }
}
