package com.xebialabs.xlrelease.risk.service

import com.xebialabs.deployit.plugin.api.reflect.{DescriptorRegistry, Type}
import com.xebialabs.xlrelease.actors.ReleaseActorService
import com.xebialabs.xlrelease.domain.Release
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.repository.Ids.ROOT_FOLDER_ID
import com.xebialabs.xlrelease.repository.ReleaseRepository
import com.xebialabs.xlrelease.risk.domain.Risk.RISK_PREFIX
import com.xebialabs.xlrelease.risk.domain.RiskProfile.{DEFAULT_RISK_PROFILE_ID, RISK_PROFILE}
import com.xebialabs.xlrelease.risk.domain.events.RiskScoreUpdated
import com.xebialabs.xlrelease.risk.domain.riskassessors.RiskAssessor
import com.xebialabs.xlrelease.risk.domain.{Risk, RiskAssessment, RiskProfile}
import com.xebialabs.xlrelease.risk.repository.RiskRepository
import com.xebialabs.xlrelease.service.FeatureService
import com.xebialabs.xlrelease.utils.Diff
import org.springframework.stereotype.Service

import scala.jdk.CollectionConverters._

@Service
class RiskService(riskRepository: RiskRepository,
                  riskProfileService: RiskProfileService,
                  releaseActorService: ReleaseActorService,
                  releaseRepository: ReleaseRepository,
                  eventBus: XLReleaseEventBus,
                  asyncRiskExecutor: AsyncRiskExecutor) extends FeatureService with RiskCalculator {

  @volatile
  private var enabled = true

  private val riskAssessorTypes: List[Type] = DescriptorRegistry.getSubtypes(Type.valueOf(classOf[RiskAssessor])).asScala
    .filter(!_.getDescriptor.isVirtual).toList

  def calculateRiskAndUpdateRelease(releaseId: String): Unit = {
    if (enabled) {
      asyncRiskExecutor.executeAsync {
        val release = releaseRepository.findById(releaseId)
        calculateRiskAndUpdateRelease(release)
      }
    }
  }

  def calculateRiskAndUpdateRelease(release: Release): Unit = {
    if (enabled) {
      asyncRiskExecutor.executeAsync {
        if (release.isPlannedOrActive && !release.isWorkflow) {
          val originalRisk = riskRepository.findByIdOrDefault(generateRiskId(release.getId))
          val updatedRisk = calculateRisk(originalRisk, release)
          updateRisk(release.getId, originalRisk, updatedRisk)
        }
      }
    }
  }

  def calculateRisk(originalRisk: Risk, release: Release): Risk = {
    if (enabled && release.isPlannedOrActive && !release.isWorkflow) {
      val riskProfileByRelease: RiskProfile = getRiskProfile(release)
      // Get all applicable risk assessors for this risk profile
      val riskAssessors: List[RiskAssessor] = riskAssessorTypes
        .filter(riskProfileByRelease.hasRiskAssessorEnabled) // filter out enabled assessors only
        .map { riskAssessorType =>
          riskAssessorType.getDescriptor.newInstance[RiskAssessor](generateAssessorId(riskAssessorType.getName))
        }
      val riskId = generateRiskId(release.getId)
      calculateRisk(riskProfileByRelease, riskAssessors, riskId, release)
    } else {
      originalRisk
    }
  }

  def getRisk(releaseId: String): Risk = {
    val riskId = generateRiskId(releaseId)
    riskRepository.findByIdOrDefault(riskId)
  }

  private def getRiskProfile(release: Release): RiskProfile = {
    Option(release.getProperty[RiskProfile](RISK_PROFILE))
      .getOrElse(riskProfileService.findByIdOrDefault(DEFAULT_RISK_PROFILE_ID))
  }

  def updateRisk(releaseId: String, original: Risk, updated: Risk): Risk = {
    if (scoreChanged(original, updated) || riskAssessmentChanged(original, updated)) {
      if (original.getRiskAssessments.isEmpty) {
        riskRepository.create(updated)
      } else {
        riskRepository.update(updated)
      }
      logger.info(
        s"""Updated risk score for release $releaseId from ${original.getScore} to ${updated.getScore} and total risk score from ${original.getTotalScore} to ${updated.getTotalScore}""".stripMargin
      )
      releaseRepository.setRiskScores(releaseId, updated.getScore, updated.getTotalScore)
      releaseActorService.updateReleaseRiskScores(releaseId, updated.getScore, updated.getTotalScore)
      eventBus.publish(RiskScoreUpdated(releaseId, Option(original), updated))
    }
    updated
  }

  private def riskAssessmentChanged(a: Risk, b: Risk): Boolean = {
    val keyMapping: RiskAssessment => String = (ra: RiskAssessment) => ra.getRiskAssessorId
    val areEqual: (RiskAssessment, RiskAssessment) => Boolean = (ra1, ra2) => {
      def safeEquals(x: Any, y: Any): Boolean = Option(x) == Option(y)

      safeEquals(ra1.getScore, ra2.getScore) &&
        safeEquals(ra1.getHeadline, ra2.getHeadline) &&
        safeEquals(ra1.getMessages, ra2.getMessages)
    }

    val diff = Diff.applyWithKeyMappingAndComparator[String, RiskAssessment](a.getRiskAssessments.asScala, b.getRiskAssessments.asScala)(keyMapping, areEqual)
    diff.updatedEntries.nonEmpty || diff.newEntries.nonEmpty || diff.deletedEntries.nonEmpty
  }

  private def scoreChanged(a: Risk, b: Risk) = (!(a.getScore == b.getScore)) || (!(a.getTotalScore == b.getTotalScore))

  private def generateRiskId(releaseId: String) = s"$releaseId/$RISK_PREFIX"

  private def generateAssessorId(assessorName: String): String = s"$ROOT_FOLDER_ID/$assessorName"

  override def serviceName(): String = "RiskService"

  override def start(): Unit = {
    this.enabled = true
  }

  override def stop(): Unit = {
    this.enabled = false
  }

  override def isRunning: Boolean = enabled

}