package com.xebialabs.xlrelease.risk.service

import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.domain.Release
import com.xebialabs.xlrelease.domain.events._
import com.xebialabs.xlrelease.events.{EventListener, Subscribe, SynchronizedSubscribe}
import com.xebialabs.xlrelease.repository.ConfigurationRepository
import com.xebialabs.xlrelease.risk.domain.RiskProfile
import com.xebialabs.xlrelease.risk.domain.events.RiskProfileUpdated
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import java.util.concurrent.Executor


@Service
@EventListener
class RiskEventHandler @Autowired()(xlrConfig: XlrConfig,
                                    configurationRepository: ConfigurationRepository,
                                    riskService: RiskService,
                                    riskCalculationBatchExecutor: Executor)
  extends Logging {

  @Subscribe
  def releaseCreated(releaseCreatedEvent: ReleaseCreatedEvent): Unit = {
    calculateRisk(releaseCreatedEvent.release)
  }

  @Subscribe
  def releaseUpdated(releaseUpdatedEvent: ReleaseUpdatedEvent): Unit = {
    calculateRisk(releaseUpdatedEvent.updated)
  }

  @Subscribe
  def releaseOverdue(releaseOverdueEvent: ReleaseOverdueEvent): Unit = {
    calculateRisk(releaseOverdueEvent.release)
  }

  @Subscribe
  def taskFailed(taskFailedEvent: TaskFailedEvent): Unit = {
    calculateRisk(taskFailedEvent.task.getRelease)
  }

  @Subscribe
  def taskCompleted(taskCompletedEvent: TaskCompletedEvent): Unit = {
    calculateRisk(taskCompletedEvent.task.getRelease)
  }

  @Subscribe
  def taskSkipped(taskSkippedEvent: TaskSkippedEvent): Unit = {
    calculateRisk(taskSkippedEvent.task.getRelease)
  }

  @Subscribe
  def taskAborted(taskAbortedEvent: TaskAbortedEvent): Unit = {
    calculateRisk(taskAbortedEvent.task.getRelease)
  }

  @Subscribe
  def taskUpdated(taskUpdatedEvent: TaskUpdatedEvent): Unit = {
    calculateRisk(taskUpdatedEvent.updated.getRelease)
  }

  @Subscribe
  def taskOverdue(taskOverdueEvent: TaskOverdueEvent): Unit = {
    calculateRisk(taskOverdueEvent.task.getRelease)
  }

  @Subscribe
  def taskDeleted(taskDeletedEvent: TaskDeletedEvent): Unit = {
    calculateRisk(taskDeletedEvent.task.getRelease)
  }

  private def calculateRisk(release: Release): Unit = {
    if (xlrConfig.isRiskCalculationEnabled && riskService.isRunning) {
      riskService.calculateRiskAndUpdateRelease(release)
    }
  }

  @SynchronizedSubscribe
  def handleRiskProfileUpdate(event: RiskProfileUpdated): Unit = {
    if (xlrConfig.isRiskCalculationEnabled) {
      val riskProfile = event.riskProfile
      riskCalculationBatchExecutor.execute(() => {
        runCalculationsForRiskProfile(riskProfile)
      })
    }
  }

  private def runCalculationsForRiskProfile(riskProfile: RiskProfile): Unit = {
    logger.debug(s"Running risk calculations after Risk profile ${riskProfile.getId} update")
    configurationRepository.getReferenceReleases(riskProfile.getId).forEach(data => {
      riskService.calculateRiskAndUpdateRelease(data.id)
    })
    logger.debug(s"Finished running risk calculations after Risk profile ${riskProfile.getId} update")
  }

}
