package com.xebialabs.xlrelease.risk.service

import com.xebialabs.deployit.plugin.api.reflect.{DescriptorRegistry, Type}
import com.xebialabs.xlrelease.domain.Release
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.repository.Ids.{ROOT_FOLDER_ID, SEPARATOR}
import com.xebialabs.xlrelease.repository.ReleaseRepository
import com.xebialabs.xlrelease.risk.domain.Risk.RISK_PREFIX
import com.xebialabs.xlrelease.risk.domain.RiskProfile.{DEFAULT_RISK_PROFILE_ID, RISK_PROFILE}
import com.xebialabs.xlrelease.risk.domain.events.RiskScoreUpdated
import com.xebialabs.xlrelease.risk.domain.riskassessors.RiskAssessor
import com.xebialabs.xlrelease.risk.domain.{Risk, RiskAssessment, RiskProfile}
import com.xebialabs.xlrelease.risk.repository.RiskRepository
import com.xebialabs.xlrelease.risk.spring.config.RiskConfiguration.RISK_CALCULATION_EXECUTOR
import com.xebialabs.xlrelease.service.FeatureService
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.context.annotation.{Scope, ScopedProxyMode}
import org.springframework.scheduling.annotation.Async
import org.springframework.stereotype.Service

import scala.jdk.CollectionConverters._

@Service
@Scope(proxyMode = ScopedProxyMode.TARGET_CLASS)
class RiskService @Autowired()(riskRepository: RiskRepository,
                               riskProfileService: RiskProfileService,
                               releaseRepository: ReleaseRepository,
                               eventBus: XLReleaseEventBus) extends FeatureService with Logging {

  @volatile
  private var enabled = true

  private val riskAssessorTypes: List[Type] = DescriptorRegistry.getSubtypes(Type.valueOf(classOf[RiskAssessor])).asScala
    .filter(!_.getDescriptor.isVirtual).toList

  def calculateRisk(release: Release): Risk = {
    logger.debug(s"Running risk calculations for release ${release.getId}")

    val risk: Risk = riskRepository.findByIdOrDefault(generateRiskId(release.getId))
    val riskProfileByRelease: RiskProfile = Option(release.getProperty[RiskProfile](RISK_PROFILE))
      .getOrElse(riskProfileService.findByIdOrDefault(DEFAULT_RISK_PROFILE_ID))

    // Get all applicable risk assessors for this risk profile
    val riskAssessors: List[RiskAssessor] = riskAssessorTypes
      .filter(riskProfileByRelease.hasRiskAssessorEnabled) // filter out enabled assessors only
      .map { riskAssessorType =>
        riskAssessorType.getDescriptor.newInstance[RiskAssessor](generateAssessorId(riskAssessorType.getName))
      }

    // Evaluate risk assessors
    val riskAssessments: List[RiskAssessment] = riskAssessors.map { riskAssessor =>
      riskAssessor.execute(release, riskProfileByRelease)
    }

    risk.setRiskAssessments(riskAssessments.asJava)

    val score: Int = riskAssessments.map(_.getScore.toInt).maxOption.getOrElse(0)
    val totalScore: Int = riskAssessments.map(_.getScore.toInt).sum

    risk.setScore(score)
    risk.setTotalScore(totalScore)

    logger.debug(s"Finished running risk calculations for release ${release.getId} with score ${risk.getScore} and total score ${risk.getTotalScore}")

    risk
  }

  @Async(RISK_CALCULATION_EXECUTOR)
  def calculateRiskAndUpdateRelease(releaseId: String): Unit = {
    val release = releaseRepository.findById(releaseId)
    calculateRiskAndUpdateRelease(release)
  }

  @Async(RISK_CALCULATION_EXECUTOR)
  def calculateRiskAndUpdateRelease(release: Release): Unit = {
    if (release.isPlannedOrActive) {
      val risk = calculateRisk(release)
      if (enabled) updateRisk(release.getId, risk)
    }
  }

  private def updateRisk(releaseId: String, updated: Risk) = {
    logger.debug(s"UpdateRisk($releaseId)")
    val riskId: String = s"$releaseId$SEPARATOR$RISK_PREFIX"
    val maybeOriginal: Option[Risk] = riskRepository.read(riskId)

    val doUpdate: Boolean = maybeOriginal match {
      case None =>
        riskRepository.create(updated)
        logger.info(s"Created risk score for release $releaseId: ${updated.getScore} and total risk score ${updated.getTotalScore}")
        true
      case Some(risk: Risk) =>
        val original: Risk = risk
        // update regardless of score, because Risk contains RiskAssessment messages as well
        riskRepository.update(updated)
        if (scoreChanged(original, updated)) {
          logger.info(
            s"""Updated risk score for release $releaseId from ${original.getScore} to ${updated.getScore}
               | and total risk score from ${original.getTotalScore} to ${updated.getTotalScore}""".stripMargin
          )
          true
        } else {
          false
        }
    }

    if (doUpdate) {
      releaseRepository.setRiskScores(releaseId, updated.getScore, updated.getTotalScore)
      eventBus.publish(RiskScoreUpdated(releaseId, maybeOriginal, updated))
    }
  }

  private def scoreChanged(a: Risk, b: Risk) = (!(a.getScore == b.getScore)) || (!(a.getTotalScore == b.getTotalScore))

  private def generateRiskId(releaseId: String) = s"$releaseId/$RISK_PREFIX"

  private def generateAssessorId(assessorName: String): String = s"$ROOT_FOLDER_ID/$assessorName"

  override def name(): String = "RiskService"

  override def enable(): Unit = {
    this.enabled = true
  }

  override def disable(): Unit = {
    this.enabled = false
  }

  override def isEnabled: Boolean = enabled

}
