package com.xebialabs.xlrelease.risk.api.v1.impl;

import java.text.MessageFormat;
import java.util.List;
import java.util.stream.Collectors;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.xlrelease.api.v1.RiskApi;
import com.xebialabs.xlrelease.domain.Release;
import com.xebialabs.xlrelease.repository.ConfigurationRepository;
import com.xebialabs.xlrelease.repository.Ids;
import com.xebialabs.xlrelease.risk.configuration.RiskGlobalThresholds;
import com.xebialabs.xlrelease.risk.domain.Risk;
import com.xebialabs.xlrelease.risk.domain.RiskProfile;
import com.xebialabs.xlrelease.risk.domain.riskassessors.RiskAssessor;
import com.xebialabs.xlrelease.risk.repository.RiskRepository;
import com.xebialabs.xlrelease.risk.service.RiskProfileService;
import com.xebialabs.xlrelease.security.PermissionChecker;

import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.security.permission.PlatformPermissions.ADMIN;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.EDIT_RISK_PROFILE;
import static java.lang.Integer.parseInt;

@Controller
public class RiskApiImpl implements RiskApi {

    private PermissionChecker permissions;

    private RiskProfileService riskProfileService;

    private ConfigurationRepository configurationRepository;

    private RiskRepository riskRepository;

    @Autowired
    public RiskApiImpl(PermissionChecker permissions, RiskRepository riskRepository, ConfigurationRepository configurationRepository, final RiskProfileService riskProfileService) {
        this.permissions = permissions;
        this.configurationRepository = configurationRepository;
        this.riskRepository = riskRepository;
        this.riskProfileService = riskProfileService;
    }

    @Override
    public Risk getRisk(String riskId) {
        permissions.checkView(Ids.releaseIdFrom(riskId));
        return riskRepository.findByIdOrDefault(riskId);
    }

    @Override
    public Risk getRisk(final Release release) {
        return getRisk(release.getId() + "/Risk");
    }

    @Override
    public RiskGlobalThresholds getRiskGlobalThresholds() {
        return configurationRepository.read(RiskGlobalThresholds.RISK_GLOBAL_THRESHOLD_SETTINGS_ID);
    }

    @Override
    public RiskGlobalThresholds updateRiskGlobalThresholds(final RiskGlobalThresholds thresholds) {
        permissions.check(ADMIN);
        configurationRepository.update(thresholds);
        return configurationRepository.read(thresholds.getId());
    }

    @Override
    public List<RiskProfile> getRiskProfiles() {
        return riskProfileService.findAll();
    }

    @Override
    public RiskProfile getRiskProfile(String riskProfileId) {
        return riskProfileService.findByIdOrDefault(riskProfileId);
    }

    @Override
    public RiskProfile getRiskProfileByTitle(String title) {
        return riskProfileService.findByTitle(title);
    }

    @Override
    public RiskProfile updateRiskProfile(String riskProfileId, RiskProfile riskProfile) {
        permissions.check(EDIT_RISK_PROFILE);
        validateValues(riskProfile);
        return riskProfileService.update(riskProfile);
    }

    @Override
    public RiskProfile updateRiskProfile(final RiskProfile riskProfile) {
        return updateRiskProfile(riskProfile.getId(), riskProfile);
    }

    @Override
    public RiskProfile createRiskProfile(RiskProfile riskProfile) {
        permissions.check(EDIT_RISK_PROFILE);
        validateValues(riskProfile);
        return riskProfileService.create(riskProfile);
    }

    @Override
    public void deleteRiskProfile(String riskProfileId) {
        permissions.check(EDIT_RISK_PROFILE);
        riskProfileService.delete(riskProfileId);
    }

    @Override
    public void deleteRiskProfile(final RiskProfile riskProfile) {
        deleteRiskProfile(riskProfile.getId());
    }

    @Override
    public RiskProfile copyRiskProfile(String riskProfileId) {
        permissions.check(EDIT_RISK_PROFILE);
        return riskProfileService.copy(riskProfileId);
    }

    @Override
    public RiskProfile copyRiskProfile(final RiskProfile riskProfile) {
        return copyRiskProfile(riskProfile.getId());
    }

    @Override
    public List<RiskAssessor> getAllRiskAssessors() {
        return DescriptorRegistry.getSubtypes(Type.valueOf(RiskAssessor.class))
                .stream()
                .filter(type -> !type.getDescriptor().isVirtual())
                .map(type -> type.getDescriptor().<RiskAssessor>newInstance(generateAssessorId(type.getName())))
                .collect(Collectors.toList());
    }

    private String generateAssessorId(String name) {
        return MessageFormat.format("{0}/{1}", Ids.ROOT_FOLDER_ID, name);
    }

    private void validateValues(RiskProfile riskProfile) {
        riskProfile.getRiskProfileAssessors().forEach((profile,score) -> {
            checkArgument(score.chars().allMatch(Character::isDigit) && !(parseInt(score) < 0) && !(parseInt(score) > 100),
                    String.format("Incorrect value %s for risk assessor %s. Value must be an integer between 0 and 100", score, profile));
        });
    }
}
