package com.xebialabs.xlrelease.events.handler

import com.xebialabs.deployit.repository.ItemInUseException
import com.xebialabs.xlrelease.actors.ReleaseGroupActorService
import com.xebialabs.xlrelease.domain.Release
import com.xebialabs.xlrelease.domain.events._
import com.xebialabs.xlrelease.events.{EventListener, SynchronizedSubscribe}
import com.xebialabs.xlrelease.json.JsonUtils
import com.xebialabs.xlrelease.risk.domain.events.RiskScoreUpdated
import com.xebialabs.xlrelease.service.ReleaseGroupService
import com.xebialabs.xlrelease.validation.{FolderOperationReferenceCategory, FolderOperationValidationError, FolderValidationErrors}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import scala.collection.mutable.ListBuffer


@Component
@EventListener
class ReleaseGroupEventHandler @Autowired()(releaseGroupActorService: ReleaseGroupActorService,
                                            releaseGroupService: ReleaseGroupService) {
  @SynchronizedSubscribe
  def onReleaseExecution(event: ReleaseExecutionEvent): Unit = triggerStatusCalculation(event.release)

  @SynchronizedSubscribe
  def onRiskScoreUpdated(event: RiskScoreUpdated): Unit = {
    releaseGroupService.findGroupsReferencingRelease(event.releaseId).foreach(releaseGroupActorService.updateGroupRisk)
  }

  @SynchronizedSubscribe
  def onFolderDeleting(action: FolderDeletingAction): Unit = {
    val groups = releaseGroupService.findActiveGroupsReferencingFolder(action.folderId)
    if (groups.nonEmpty) {
      val validationMessages = ListBuffer.empty[FolderOperationValidationError]
      groups.take(20).map(group => {
        val error = FolderOperationValidationError(
          group.groupId,
          group.title,
          category = FolderOperationReferenceCategory.RELEASE_GROUP_CATEGORY,
          message = s"The folder or its subfolders contain releases referenced by active release group '${group.title}'"
        )
        validationMessages += error
      })

      val errors = FolderValidationErrors("You cannot delete the folder", validationMessages.toList)
      val json = JsonUtils.objectMapper.writeValueAsString(errors)
      throw new ItemInUseException(json)
    }
  }

  private def triggerStatusCalculation(release: Release): Unit = {
    releaseGroupService.findGroupsReferencingRelease(release.getId).foreach(releaseGroupActorService.updateGroupStatus)
  }
}
