package com.xebialabs.xlrelease.repository.sql.persistence

import com.xebialabs.deployit.security.permission.Permission
import com.xebialabs.xlrelease.api.v1.forms.ReleaseGroupOrderMode
import com.xebialabs.xlrelease.api.v1.forms.ReleaseGroupOrderMode.{END_DATE, RISK, START_DATE}
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.{Sql, SqlBuilder}
import com.xebialabs.xlrelease.domain.group.ReleaseGroupStatus
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.CiId
import com.xebialabs.xlrelease.repository.sql.persistence.ReleaseGroupSchema.RELEASE_GROUPS
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.FOLDERS
import com.xebialabs.xlrelease.utils.FolderId

object ReleaseGroupSqlBuilder {
  val alias: String = "rg"
  val folderAlias: String = "f"

  val STMT_RELEASE_GROUP_SELECT: String =
    s"""|SELECT $alias.${RELEASE_GROUPS.CI_UID}, $folderAlias.${FOLDERS.FOLDER_PATH}, $folderAlias.${FOLDERS.FOLDER_ID}, $alias.${RELEASE_GROUPS.CONTENT}
        | FROM ${RELEASE_GROUPS.TABLE} $alias
        | INNER JOIN ${FOLDERS.TABLE} $folderAlias ON $alias.${RELEASE_GROUPS.FOLDER_UID} = $folderAlias.${FOLDERS.CI_UID}""".stripMargin

}

class ReleaseGroupSqlBuilder(implicit dialect: Dialect) extends SqlBuilder[ReleaseGroupSqlBuilder] with SecurableSqlBuilder {

  import ReleaseGroupSqlBuilder._

  def select(): ReleaseGroupSqlBuilder = {
    super.select(STMT_RELEASE_GROUP_SELECT)
  }

  def withTitleLike(title: String): ReleaseGroupSqlBuilder = {
    like(s"$alias.${RELEASE_GROUPS.TITLE}", title)
  }

  def withOneOfStatuses(statuses: Seq[ReleaseGroupStatus]): ReleaseGroupSqlBuilder = {
    if (statuses.nonEmpty) {
      conditions += Sql(s"$alias.${RELEASE_GROUPS.STATUS} IN (${statuses.map(_ => "?").mkString(",")})", statuses.map(_.value()))
    }
    this
  }

  def withFolder(folderId: CiId): ReleaseGroupSqlBuilder = {
    if (folderId != null && !folderId.isEmpty) {
      conditions += Sql(s"$folderAlias.${FOLDERS.FOLDER_ID} = ?", Seq(FolderId(folderId).id))
    }
    this
  }

  def withPermission(permission: Option[Permission], principals: Iterable[String], roleIds: Iterable[String]): ReleaseGroupSqlBuilder = {
    if (permission.isDefined) {
      conditions += Sql(
        s"""$folderAlias.${FOLDERS.SECURITY_UID} IN (
           |  ${selectCiIdsWithPermission(principals, roleIds)}
           |)
      """.stripMargin,
        Seq(permission.get.getPermissionName) ++ principals.map(_.toLowerCase) ++ roleIds
      )
    }
    this
  }

  def orderBy(order: ReleaseGroupOrderMode): ReleaseGroupSqlBuilder = {
    order match {
      case START_DATE =>
        orderBy(s"$alias.${RELEASE_GROUPS.START_DATE} DESC")
      case END_DATE =>
        orderBy(s"$alias.${RELEASE_GROUPS.END_DATE} DESC")
      case RISK =>
        orderBy(s"$alias.${RELEASE_GROUPS.RISK_SCORE} DESC")
    }
    orderBy(s"$alias.${RELEASE_GROUPS.TITLE} ASC")
    this
  }

  override def newInstance: ReleaseGroupSqlBuilder = new ReleaseGroupSqlBuilder()

}
